#!/usr/bin/env python3

#****************************************************************************
# colorset.py, provides storage/retrieval and dialogs for system colors
#
# ConvertAll, a units conversion program
# Copyright (C) 2019, Douglas W. Bell
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License, either Version 2 or any later
# version.  This program is distributed in the hope that it will be useful,
# but WITTHOUT ANY WARRANTY.  See the included LICENSE file for details.
#*****************************************************************************

import enum
from collections import OrderedDict
from PyQt5.QtCore import pyqtSignal, Qt, QEvent, QObject
from PyQt5.QtGui import QColor, QFontMetrics, QPalette, QPixmap
from PyQt5.QtWidgets import (QApplication, QColorDialog, QComboBox, QDialog,
                             QFrame, QGroupBox, QHBoxLayout, QLabel,
                             QGridLayout, QPushButton, QVBoxLayout, qApp)

roles = OrderedDict([('Window', _('Dialog background color')),
                     ('WindowText', _('Dialog text color')),
                     ('Base', _('Text widget background color')),
                     ('Text', _('Text widget foreground color')),
                     ('Highlight', _('Selected item background color')),
                     ('HighlightedText', _('Selected item text color')),
                     ('Button', _('Button background color')),
                     ('ButtonText', _('Button text color')),
                     ('Text-Disabled', _('Disabled text foreground color')),
                     ('ButtonText-Disabled', _('Disabled button text color'))])

ThemeSetting = enum.IntEnum('ThemeSetting', 'system dark custom')

darkColors = {'Window': '#353535', 'WindowText': '#ffffff',
              'Base': '#191919', 'Text': '#ffffff',
              'Highlight': '#2a82da', 'HighlightedText': '#000000',
              'Button': '#353535', 'ButtonText': '#ffffff',
              'Text-Disabled': '#808080', 'ButtonText-Disabled': '#808080'}

class ColorSet:
    """Stores color settings and provides dialogs for user changes.
    """
    def __init__(self, option):
        self.option = option
        self.sysPalette = QApplication.palette()
        self.colors = [Color(roleKey) for roleKey in roles.keys()]
        self.theme = ThemeSetting[self.option.strData('ColorTheme')]
        for color in self.colors:
            color.colorChanged.connect(self.setCustomTheme)
            color.setFromPalette(self.sysPalette)
            if self.theme == ThemeSetting.dark:
                color.setFromTheme(darkColors)
            elif self.theme == ThemeSetting.custom:
                color.setFromOption(self.option)

    def setAppColors(self):
        """Set application to current colors.
        """
        newPalette = QApplication.palette()
        for color in self.colors:
            color.updatePalette(newPalette)
        qApp.setPalette(newPalette)


    def showDialog(self, parent):
        """Show a dialog for user color changes.

        Return True if changes were made.
        """
        dialog = QDialog(parent)
        dialog.setWindowFlags(Qt.Dialog | Qt.WindowTitleHint |
                              Qt.WindowSystemMenuHint)
        dialog.setWindowTitle(_('Color Settings'))
        topLayout = QVBoxLayout(dialog)
        dialog.setLayout(topLayout)
        themeBox = QGroupBox(_('Color Theme'), dialog)
        topLayout.addWidget(themeBox)
        themeLayout = QVBoxLayout(themeBox)
        self.themeControl = QComboBox(dialog)
        self.themeControl.addItem(_('Default system theme'),
                                  ThemeSetting.system)
        self.themeControl.addItem(_('Dark theme'), ThemeSetting.dark)
        self.themeControl.addItem(_('Custom theme'), ThemeSetting.custom)
        self.themeControl.setCurrentIndex(self.themeControl.
                                          findData(self.theme))
        self.themeControl.currentIndexChanged.connect(self.updateThemeSetting)
        themeLayout.addWidget(self.themeControl)
        self.groupBox = QGroupBox(dialog)
        self.setBoxTitle()
        topLayout.addWidget(self.groupBox)
        gridLayout = QGridLayout(self.groupBox)
        row = 0
        for color in self.colors:
            gridLayout.addWidget(color.getLabel(), row, 0)
            gridLayout.addWidget(color.getSwatch(), row, 1)
            row += 1
        ctrlLayout = QHBoxLayout()
        topLayout.addLayout(ctrlLayout)
        ctrlLayout.addStretch(0)
        okButton = QPushButton(_('&OK'), dialog)
        ctrlLayout.addWidget(okButton)
        okButton.clicked.connect(dialog.accept)
        cancelButton = QPushButton(_('&Cancel'), dialog)
        ctrlLayout.addWidget(cancelButton)
        cancelButton.clicked.connect(dialog.reject)
        if dialog.exec_() == QDialog.Accepted:
            self.theme = ThemeSetting(self.themeControl.currentData())
            self.option.changeData('ColorTheme', self.theme.name, True)
            if self.theme == ThemeSetting.system:
                qApp.setPalette(self.sysPalette)
            else:   # dark theme or custom
                if self.theme == ThemeSetting.custom:
                    for color in self.colors:
                        color.updateOption(self.option)
                self.setAppColors()
        else:
            for color in self.colors:
                color.setFromPalette(self.sysPalette)
                if self.theme == ThemeSetting.dark:
                    color.setFromTheme(darkColors)
                elif self.theme == ThemeSetting.custom:
                    color.setFromOption(self.option)

    def setBoxTitle(self):
        """Set title of group box to standard or custom.
        """
        if self.themeControl.currentData() == ThemeSetting.custom:
            title = _('Custom Colors')
        else:
            title = _('Theme Colors')
        self.groupBox.setTitle(title)

    def updateThemeSetting(self):
        """Update the colors based on a theme control change.
        """
        if self.themeControl.currentData() == ThemeSetting.system:
            for color in self.colors:
                color.setFromPalette(self.sysPalette)
                color.changeSwatchColor()
        elif self.themeControl.currentData() == ThemeSetting.dark:
            for color in self.colors:
                color.setFromTheme(darkColors)
                color.changeSwatchColor()
        else:
            for color in self.colors:
                color.setFromOption(self.option)
                color.changeSwatchColor()
        self.setBoxTitle()

    def setCustomTheme(self):
        """Set to custom theme setting after user color change.
        """
        if self.themeControl.currentData != ThemeSetting.custom:
            self.themeControl.blockSignals(True)
            self.themeControl.setCurrentIndex(2)
            self.themeControl.blockSignals(False)
            self.setBoxTitle()


class Color(QObject):
    """Stores a single color setting for a role.
    """
    colorChanged = pyqtSignal()
    def __init__(self, roleKey, parent=None):
        super().__init__(parent)
        self.roleKey = roleKey
        if '-' in roleKey:
            roleStr, groupStr = roleKey.split('-')
            self.group = eval('QPalette.' + groupStr)
        else:
            roleStr = roleKey
            self.group = None
        self.role = eval('QPalette.' + roleStr)
        self.currentColor = None
        self.swatch = None

    def setFromPalette(self, palette):
        """Set the color based on the given palette.
        """
        if self.group:
            self.currentColor = palette.color(self.group, self.role)
        else:
            self.currentColor = palette.color(self.role)

    def setFromOption(self, option):
        """Set color based on the option setting.
        """
        colorStr = '#' + option.strData(self.roleKey + 'Color', True)
        color = QColor(colorStr)
        if color.isValid():
            self.currentColor = color

    def setFromTheme(self, theme):
        """Set color based on the given theme dictionary.
        """
        self.currentColor = QColor(theme[self.roleKey])

    def updateOption(self, option):
        """Set the option to the current color.
        """
        if self.currentColor:
            colorStr = self.currentColor.name().lstrip('#')
            option.changeData(self.roleKey + 'Color', colorStr, True)

    def updatePalette(self, palette):
        """Set the role in the given palette to the current color.
        """
        if self.group:
            palette.setColor(self.group, self.role, self.currentColor)
        else:
            palette.setColor(self.role, self.currentColor)

    def getLabel(self):
        """Return a label for this role in a dialog.
        """
        return QLabel(roles[self.roleKey])

    def getSwatch(self):
        """Return a label color swatch with the current color.
        """
        self.swatch = QLabel()
        self.changeSwatchColor()
        self.swatch.setFrameStyle(QFrame.Panel | QFrame.Raised)
        self.swatch.setLineWidth(3)
        self.swatch.installEventFilter(self)
        return self.swatch

    def changeSwatchColor(self):
        """Set swatch to currentColor.
        """
        height = QFontMetrics(self.swatch.font()).height()
        pixmap = QPixmap(3 * height, height)
        pixmap.fill(self.currentColor)
        self.swatch.setPixmap(pixmap)

    def eventFilter(self, obj, event):
        """Handle mouse clicks on swatches.
        """
        if obj == self.swatch and event.type() == QEvent.MouseButtonRelease:
            color = QColorDialog.getColor(self.currentColor,
                                          QApplication.activeWindow(),
                                          _('Select {0} color').
                                          format(self.roleKey))
            if color.isValid() and color != self.currentColor:
                self.currentColor = color
                self.changeSwatchColor()
                self.colorChanged.emit()
            return True
        return False
