/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/macros/relation_modifier.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref DIAGNOSTICS_RELATION
 *
 * $Id: relation_modifier.t.cpp,v 1.2 2005/06/23 09:54:22 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @todo add testcases
 */

#include <diagnostics/unittest.hpp>

// component
#include <diagnostics/macros/relation_modifier.hpp>

// contains the component again -- but also macros required for testing.
#include <diagnostics/annotations.hpp>

// backdoor
#include <diagnostics/unittest/test_system/current_test_logger.hpp>

// used component
#include <diagnostics/frame/logging_config.hpp>
#include <diagnostics/util/to_string.hpp>

// test support
#include <diagnostics/util/dummy_logger.ts.hpp>
#include <diagnostics/util/assert_record.ts.hpp>

#include <string>

#define TEST_COMPONENT_NAME relation_modifier
#define TEST_COMPONENT_NAMESPACE diagnostics


DIAGNOSTICS_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;
using namespace unittest;
using ::diagnostics::unittest::internal::Current_Test_Logger;


/**
 * @brief test a success and a failing assert. Check the Exception and
 * the log message.
 */ 
void test_successful_and_failing_assert(Test_Data & data)
{
    Dummy_Logger * logger=new Dummy_Logger; 
    Dummy_Logger::Records_t & records(logger->records()); 
    typedef Dummy_Logger::Records_t::iterator Iter_t;
    Logging_Config::register_logger(logger);

	int min(0);
	int max(5);
	::std::string const recrod_what_base("CONDITION=\"((min) <= (max))\" EXCEPTION=\"Test_Exception\" WHAT=\"");


	TEST_ASSERT_RELATION(records.size(),==,1);

	// no error 0<=5 //////////////////////////////////////////////////
	TEST_EXCEPTIONLESS_BLOCK_ENTER;
	TEST_NO_IMPLICIT_LOGGING_ENTER;
	DIAGNOSTICS_RELATION(DIAGNOSTICS_PROD_ASSERT1,Test_Exception,min,<=,max);
	TEST_NO_IMPLICIT_LOGGING_EXIT;
	TEST_EXCEPTIONLESS_BLOCK_EXIT;

	TEST_ASSERT_RELATION(records.size(),==,1);

	// error: 10<=5 ///////////////////////////////////////////////////
	min=10;
	::std::string const e_10_le_5("(min)<=(max) EVALUATES TO (10)<=(5)");
	::std::string const r_10_le_5(recrod_what_base+e_10_le_5+"\"");

	TEST_THROWING_BLOCK_ENTER;
	TEST_NO_IMPLICIT_LOGGING_ENTER;
	DIAGNOSTICS_RELATION(DIAGNOSTICS_PROD_ASSERT1,Test_Exception,min,<=,max);
	TEST_NO_IMPLICIT_LOGGING_EXIT;
	TEST_THROWING_BLOCK_EXIT1(Test_Exception,e_10_le_5.c_str());

	TEST_ASSERT_RELATION(records.size(),==,2);
	

	// check all records //////////////////////////////////////////////

    Iter_t current(records.begin()); 
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN);
    ++current;
	TEST_ASSERT_RECORD1(LEVEL_PROD,TYPE_FAILED_ASSERTION,0,r_10_le_5);
}

/**
 * @brief test the number of evaluations of the arguments in case of
 * an error. Right now, they are evaluated three times; it should be
 * once.
 */ 
void test_number_of_evaluations(Test_Data & data)
{
    Dummy_Logger * logger=new Dummy_Logger; 
    Dummy_Logger::Records_t & records(logger->records()); 
    typedef Dummy_Logger::Records_t::iterator Iter_t;
    Logging_Config::register_logger(logger);

	::std::string const recrod_what_base("CONDITION=\"((eval1++) <= (eval2++))\" EXCEPTION=\"Test_Exception\" WHAT=\"");

	int eval1(0);
	int eval2(5);

	TEST_ASSERT_RELATION(records.size(),==,1);

	// no error 0<=5 //////////////////////////////////////////////////
	TEST_EXCEPTIONLESS_BLOCK_ENTER;
	TEST_NO_IMPLICIT_LOGGING_ENTER;
	DIAGNOSTICS_RELATION(DIAGNOSTICS_PROD_ASSERT1,Test_Exception,eval1++,<=,eval2++);
	TEST_NO_IMPLICIT_LOGGING_EXIT;
	TEST_EXCEPTIONLESS_BLOCK_EXIT;
	
	TEST_ASSERT_RELATION(eval1,==,1);
	TEST_ASSERT_RELATION(eval2,==,6);

	TEST_ASSERT_RELATION(records.size(),==,1);

	// error: 10<=5 ///////////////////////////////////////////////////
	eval1=10;
	eval2=5;
	::std::string const e_10_le_5("(eval1++)<=(eval2++) EVALUATES TO (10)<=(5)");
	::std::string const e_11_le_6("(eval1++)<=(eval2++) EVALUATES TO (11)<=(6)");
	::std::string const e_12_le_7("(eval1++)<=(eval2++) EVALUATES TO (12)<=(7)");
	::std::string const r_11_le_6(recrod_what_base+e_11_le_6+"\"");

	TEST_THROWING_BLOCK_ENTER;
	TEST_NO_IMPLICIT_LOGGING_ENTER;
	DIAGNOSTICS_RELATION(DIAGNOSTICS_PROD_ASSERT1,Test_Exception,eval1++,<=,eval2++);
	TEST_NO_IMPLICIT_LOGGING_EXIT;
	TEST_THROWING_BLOCK_EXIT1(Test_Exception,e_12_le_7.c_str()); // evaluated TWICE!

	TEST_ASSERT_RELATION(eval1,==,13); // that's the BUG! -- evaluated THRICE!
	TEST_ASSERT_RELATION(eval2,==,8);

	TEST_ASSERT_RELATION(records.size(),==,2);

	// check all records //////////////////////////////////////////////

    Iter_t current(records.begin()); 
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN);
    ++current;
	// LOG MSG DIFFERS FROM EXCEPTION WHAT!!
	TEST_ASSERT_RECORD1(LEVEL_PROD,TYPE_FAILED_ASSERTION,0,r_11_le_6); 
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&test_successful_and_failing_assert,LEVEL_PROD);
TEST_NORMAL_CASE(&test_number_of_evaluations,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;

// vim:ts=4:sw=4
