/* refcount.hh
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2007-2008,2012 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef AGUIX_REFCOUNT_HH
#define AGUIX_REFCOUNT_HH

/**
 * This class implements a simple reference counting
 * Basically it can be used similar to shared_ptr from boost
 * or vice versa
 */
template<typename _Tp> class RefCount
{
#ifdef MEMBER_TEMPLATE_FRIENDS
private:
    template<class _Tp1> friend class RefCount;
#else
public:
#endif
    _Tp *_val;
    int *_refcount;
public:
    // The pointed-to type.
    typedef _Tp element_type;

    RefCount()
    {
        _val = NULL;
        _refcount = NULL;
    };
    RefCount( element_type *y )
    {
        _val = y;
        _refcount = new int( 1 );
    };
    RefCount( const RefCount &other )
    {
        _val = other._val;
        _refcount = other._refcount;
        if ( _refcount != NULL ) {
            (*_refcount)++;
        }
    }
    template<typename _Tp1> RefCount( const RefCount<_Tp1> &other )
    {
        _val = other._val;
        _refcount = other._refcount;
        if ( _refcount != NULL ) {
            (*_refcount)++;
        }
    }
    ~RefCount()
    {
        unref();
    };

    RefCount &operator=( const RefCount<_Tp> &other )
    {
        if ( &other != this ) {
            unref();
            _val = other._val;
            _refcount = other._refcount;
            if ( _refcount != NULL ) {
                (*_refcount)++;
            }
        }
        return *this;
    }

    template<typename _Tp1> RefCount &operator=( const RefCount<_Tp1> &other )
    {
        if ( _val != other._val ) {
            unref();
            _val = other._val;
            _refcount = other._refcount;
            if ( _refcount != NULL ) {
                (*_refcount)++;
            }
        }
        return *this;
    }

    element_type &operator*() const throw()
    {
        return *_val;
    }
    element_type *operator->() const throw()
    {
        return _val;
    }

    element_type *getVal() const throw()
    {
        return _val;
    }

    element_type *get() const throw()
    {
        return _val;
    }

    int getNumberOfReferences() const
    {
        if ( _refcount == NULL ) return 0;

        return *_refcount;
    }

protected:
    int *getRefCount() const throw()
    {
        return _refcount;
    }
    
private:
    void unref()
    {
        if ( _refcount == NULL ) return;
        
        if ( --(*_refcount) == 0 ) {
            if ( _val != NULL )
                delete _val;
            _val = NULL;
            delete _refcount;
            _refcount = NULL;
        }
    }

};

#endif
