/* --------------------------------------------------------------------------
 * driver_mixer_alsa.c
 * This module contains the low level driver to control the ALSA mixer. The
 * low level driver overloads methods of class mixer.
 *
 * Copyright 2006 Matthias Grimm <matthiasgrimm@user.sourceforge.net>
 * Copyright 2004 John Steele Scott <toojays@toojays.net>
 * Copyright 2002 Matthias Grimm <matthiasgrimm@user.sourceforge.net>
 * Copyright 1999-2000 by Jaroslav Kysela <perex@suse.cz>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <alsa/asoundlib.h>

#include <math.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "class_config.h"
#include "class_mixer.h"
#include "driver_mixer_alsa.h"
#include "debug.h"

#define SECTION "MODULE MIXER ALSA"

struct driver_alsamixer {
	char *card;	 /* name of the soundcard (usually "default" is fine) */
	GString *channels;	/* channel names to use */
	struct {
		unsigned int init_complete:1;
		unsigned int mute:1;
		unsigned int:0;
	} flags;
	snd_mixer_t *mixer;
	snd_mixer_elem_t **elements; /* points to a NULL terminated list, first is master */
} moddata;

/** free currently open alsa card ressources
 */
void
driver_mixer_alsa_free ()
{
	struct driver_alsamixer *base = &moddata;

	if (base->elements) {
		free (base->elements);
		base->elements = NULL;
	}

	if (base->mixer) {
		snd_mixer_detach (base->mixer, base->card);
		snd_mixer_close (base->mixer);
		base->mixer = NULL;
	}
	
	base->flags.init_complete = 0;
}

void
driver_mixer_alsa_exit ()
{
	struct driver_alsamixer *base = &moddata;

	driver_mixer_alsa_free ();
	
	g_free (base->card);
	g_string_free (base->channels, TRUE);
}

/* initialise the list of elements, based on the "channels" string */
static int
alsamixer_setup_elements_internal (char *channels, snd_mixer_selem_id_t *sid)
{
	struct driver_alsamixer *base = &moddata;
	char **element;
	int n, m;

	/* get rid of old list */
	if (base->elements) {
		free (base->elements);
		base->elements = NULL;
	}
	base->flags.init_complete = 0;

	element = g_strsplit (channels, ",", 0);
	for (n=0,m=0; element[n] != NULL; n++) {
		g_strstrip (element[n]);

		/* ensure we have enough memory to store pointer + terminator */
		base->elements = realloc (base->elements, (m+2)*sizeof (snd_mixer_elem_t *));
		if (base->elements == NULL) {
			print_msg (PBB_ERR, _("Memory allocation failed.\n"));
			return -1;
		}

		/* find element */
		for (base->elements[m] = snd_mixer_first_elem (base->mixer);
		     base->elements[m] != NULL;
		     base->elements[m] = snd_mixer_elem_next (base->elements[m])) {
			snd_mixer_selem_get_id (base->elements[m], sid);
			if (strcasecmp (snd_mixer_selem_id_get_name (sid), element[n]) == 0)
				break;
		}
			
		if (base->elements[m] == NULL) {
			/* report any element we can't find */
			print_msg (PBB_WARN, _("Card '%s' has no '%s' element.\n"), base->card, element[n]);
		} else 
			m++;	
	}

	/* terminate list */
	base->elements[m] = NULL;
	g_strfreev (element);

	/* check that we have a master element */
	if (!base->elements[0]) {
		print_msg (PBB_WARN, _("Option 'ALSA_Elements' contains no valid elements\n"));
		return -1;
	}

	/* check that master element has a playback volume */
	if (!snd_mixer_selem_has_playback_volume (base->elements[0])) {
		snd_mixer_selem_get_id (base->elements[0], sid);
		print_msg (PBB_WARN, _("Mixer element '%s' has no playback volume control.\n"), snd_mixer_selem_id_get_name (sid));
		return -1;
	}

	base->flags.init_complete = 1;
	return 0;
}

/* wrapper arround alsamixer_setup_elements_internal so we can be sure to free the sid. */
static int
alsamixer_setup_elements (char *channels)
{
	snd_mixer_selem_id_t *sid;
	int err, result;

	/* create a selem_id to extract names from */
	if ((err = snd_mixer_selem_id_malloc (&sid))){
		print_msg (PBB_ERR, _("Memory allocation failed: %s\n"), snd_strerror (err));
		return -1;
	}

	result = alsamixer_setup_elements_internal (channels, sid);
	snd_mixer_selem_id_free (sid);
	return result;
}

static int
alsamixer_finish_init ()
{
	struct driver_alsamixer *base = &moddata;
	snd_mixer_t *mixer;
	int err;

	/* setup mixer */
	if ((err = snd_mixer_open (&mixer, 0)) < 0) {
		print_msg (PBB_ERR, _("Can't open card '%s': %s\n"), base->card, snd_strerror (err));
		return -1;
	}
	if ((err = snd_mixer_attach (mixer, base->card)) < 0) {
		print_msg (PBB_ERR, _("Can't attach card '%s': %s\n"), base->card, snd_strerror (err));
		snd_mixer_close (mixer);
		return -1;
	}
	if ((err = snd_mixer_selem_register (mixer, NULL, NULL)) < 0) {
		print_msg (PBB_ERR, _("Can't register mixer: %s\n"), snd_strerror (err));
		snd_mixer_detach (mixer, base->card);
		snd_mixer_close (mixer);
		return -1;
	}
	
	if ((err = snd_mixer_load (mixer)) < 0) {
		print_msg (PBB_ERR, _("Can't load card '%s': %s\n"), base->card, snd_strerror (err));
		snd_mixer_detach (mixer, base->card);
		snd_mixer_close (mixer);
		return -1;
	}

	base->mixer = mixer;
	return alsamixer_setup_elements (base->channels->str);
}

/* Converts an ALSA volume to a number in the range [0, +/-100]. This function
 * assumes that the mixer has already been setup successfully.
 */
static int
alsamixer_vol_to_percentage (snd_mixer_elem_t * elem, long volume)
{
	long volmin, volmax;
	int pct;

	snd_mixer_selem_get_playback_volume_range (elem, &volmin, &volmax);
	pct = volmax == volmin ? 0 : rint(volume*100.0/(volmax-volmin));
	if (pct > 100) pct = 100;
	if (pct < 0)   pct = 0;
	return pct;
}

/* Converts a number in the range [0, +/-100] to an ALSA volume. This function
 * assumes that the mixer has already been setup successfully.
 */
static long
alsamixer_percentage_to_vol (snd_mixer_elem_t * elem, int pct)
{
	long volmin, volmax;
	
	snd_mixer_selem_get_playback_volume_range (elem, &volmin, &volmax);
	return rint((volmax-volmin)*pct/100.0);
}

/* Saves the current mute status in base-flags.>mute, and returns the volume of the first playback channel of the master element */
static int
alsamixer_get_volume_absolute ()
{
	struct driver_alsamixer *base = &moddata;
	long volume;
	int mute;

	if (base->flags.init_complete == 0) /* is mixer setup already completed? */
		if ((alsamixer_finish_init ()) != 0) /* no, then do it now */
			return -1; /* Oops, try it again later */

	/* force simple mixer to update its values from audio driver */
	snd_mixer_handle_events (base->mixer);

	/* read values */
	snd_mixer_selem_get_playback_switch (base->elements[0], 0, &mute);
	snd_mixer_selem_get_playback_volume (base->elements[0], 0, &volume);
	base->flags.mute = !mute;
	return volume;
}

/* same as alsamixer_get_volume_absolute, but returns the volume as number between 0 and 100 */
static int 
alsamixer_get_volume_as_percentage ()
{
	struct driver_alsamixer *base = &moddata;
	int volume;

	volume = alsamixer_get_volume_absolute ();
	return volume < 0 ? volume : alsamixer_vol_to_percentage (base->elements[0], volume);
}

gboolean
alsamixer_is_muted (void)
{
	struct driver_alsamixer *base = &moddata;
	return base->flags.mute == 1 ? TRUE : FALSE;
}

int
alsamixer_get_volume (enum valueinc type)
{
	switch (type) {
		case VALUE_ABS:
			return alsamixer_get_volume_absolute ();
		case VALUE_ABS_PERCENT:
			return alsamixer_get_volume_as_percentage ();
		case VALUE_REL:
		case VALUE_REL_PERCENT:
		default:
			return -1;   /* no relative values available */
	}
}

void
alsamixer_set_volume (enum valueinc type, int volume)
{
	struct driver_alsamixer *base = &moddata;
	long master = 0, vol, min, max;
	unsigned int n;

	if (base->flags.init_complete == 0)      /* is mixer setup already completed? */
		if ((alsamixer_finish_init ()) != 0) /* no, then do it now */
			return;                         /* Oops, try it again later */

	base->flags.mute = volume ? 0 : !base->flags.mute;
	
	/* cycle through elements */
	for (n=0; base->elements[n] != NULL; n++) {
		
		/* sync playback switch */
		snd_mixer_selem_set_playback_switch_all (base->elements[n], !base->flags.mute);
		
		if (n == 0) {
			switch (type) {
			case VALUE_REL:
				snd_mixer_selem_get_playback_volume_range (base->elements[0], &min, &max);
				if (max > 100) 
					volume *= max/100 + 1;
				volume += alsamixer_get_volume_absolute ();
			case VALUE_ABS:
				master = alsamixer_vol_to_percentage (base->elements[0], volume);
				break;
			case VALUE_REL_PERCENT:
				volume += alsamixer_get_volume_as_percentage ();
			case VALUE_ABS_PERCENT:
				if (volume > 100) volume = 100;
				if (volume < 0)   volume = 0;
				master = volume;
				break;
			}
		}
		
		/* sync volume with master element, if not muted */
		if (!base->flags.mute && snd_mixer_selem_has_playback_volume (base->elements[n])) {
			vol = alsamixer_percentage_to_vol (base->elements[n], master);
			snd_mixer_selem_set_playback_volume_all (base->elements[n], vol);
		}
	}
}

void
alsamixer_set_mixer (char *card)
{
	struct driver_alsamixer *base = &moddata;
	GString *tmp;

	if (base->card)
		g_free (base->card);

	/* set the new mixer card */
	tmp = g_string_new (card);
	base->card = tmp->str;
	g_string_free (tmp, FALSE);

	/* save new mixer card to configuration file */
	config_set_string (SECTION, "Card", base->card);

	/* re-initialize the mixer with next usage */
	driver_mixer_alsa_free();
}

const char *
alsamixer_get_mixer (void)
{
	struct driver_alsamixer *base = &moddata;
	return base->card;
}

void
alsamixer_set_channels (char *channels)
{
	struct driver_alsamixer *base = &moddata;
	base->channels = g_string_assign(base->channels, channels);
	
	/* save new mixer channels to configuration file */
	config_set_string (SECTION, "Channels", base->channels->str);

	driver_mixer_alsa_free();
}

const char *
alsamixer_get_channels (void)
{
	struct driver_alsamixer *base = &moddata;
	return base->channels->str;
}

static struct driver_mixer driver_mixer_alsa = {
	.name         = N_("ALSA Mixer"),
	.get_mixer    = alsamixer_get_mixer,
	.set_mixer    = alsamixer_set_mixer,
	.get_channels = alsamixer_get_channels,
	.set_channels = alsamixer_set_channels,
	.get_volume   = alsamixer_get_volume,
	.set_volume   = alsamixer_set_volume,
	.is_muted     = alsamixer_is_muted,
	.driver_exit  = driver_mixer_alsa_exit
};

struct driver_mixer *
driver_mixer_alsa_init ()
{
	struct driver_alsamixer *base = &moddata;
	char **channels;
	int n;

	base->flags.mute	      = 0;
	base->flags.init_complete = 0;
	base->channels            = g_string_new (NULL);
	base->elements            = NULL;
	base->mixer               = NULL;

	if ((check_devorfile ("/proc/asound", TYPE_DIR)) == 0) {
		base->card = config_get_string (SECTION, "Device", "default");
		channels = config_get_strlist (SECTION, "Channels", "Master");
		for (n=0; channels[n] != NULL; n++) {
			if (n > 0)
				g_string_append (base->channels, ", ");
			g_string_append (base->channels, channels[n]);
		}
		g_strfreev (channels);
		return &driver_mixer_alsa;
	}
	return NULL;
}


