# (c) 2012-2014, Michael DeHaan <michael.dehaan@gmail.com>
#
# This file is part of Ansible
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.
from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

import sys

from ansible import constants as C

ANSIBLE_COLOR=True
if C.ANSIBLE_NOCOLOR:
    ANSIBLE_COLOR=False
elif not hasattr(sys.stdout, 'isatty') or not sys.stdout.isatty():
    ANSIBLE_COLOR=False
else:
    try:
        import curses
        curses.setupterm()
        if curses.tigetnum('colors') < 0:
            ANSIBLE_COLOR=False
    except ImportError:
        # curses library was not found
        pass
    except curses.error:
        # curses returns an error (e.g. could not find terminal)
        ANSIBLE_COLOR=False

if C.ANSIBLE_FORCE_COLOR:
        ANSIBLE_COLOR=True

# --- begin "pretty"
#
# pretty - A miniature library that provides a Python print and stdout
# wrapper that makes colored terminal text easier to use (e.g. without
# having to mess around with ANSI escape sequences). This code is public
# domain - there is no license except that you must leave this header.
#
# Copyright (C) 2008 Brian Nez <thedude at bri1 dot com>
#
# http://nezzen.net/2008/06/23/colored-text-in-python-using-ansi-escape-sequences/

codeCodes = {
    'black':     u'0;30', 'bright gray':    u'0;37',
    'blue':      u'0;34', 'white':          u'1;37',
    'green':     u'0;32', 'bright blue':    u'1;34',
    'cyan':      u'0;36', 'bright green':   u'1;32',
    'red':       u'0;31', 'bright cyan':    u'1;36',
    'purple':    u'0;35', 'bright red':     u'1;31',
    'yellow':    u'0;33', 'bright purple':  u'1;35',
    'dark gray': u'1;30', 'bright yellow':  u'1;33',
    'magenta':   u'0;35', 'bright magenta': u'1;35',
    'normal':    u'0'   ,
}

def stringc(text, color):
    """String in color."""

    if ANSIBLE_COLOR:
        return u"\033[%sm%s\033[0m" % (codeCodes[color], text)
    else:
        return text

# --- end "pretty"

def colorize(lead, num, color):
    """ Print 'lead' = 'num' in 'color' """
    if num != 0 and ANSIBLE_COLOR and color is not None:
        return u"%s%s%-15s" % (stringc(lead, color), stringc("=", color), stringc(str(num), color))
    else:
        return u"%s=%-4s" % (lead, str(num))

def hostcolor(host, stats, color=True):
    if ANSIBLE_COLOR and color:
        if stats['failures'] != 0 or stats['unreachable'] != 0:
            return u"%-37s" % stringc(host, 'red')
        elif stats['changed'] != 0:
            return u"%-37s" % stringc(host, 'yellow')
        else:
            return u"%-37s" % stringc(host, 'green')
    return u"%-26s" % host

