<?php
/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */

namespace Doctrine\ORM\Id;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\ORMException;

/**
 * Special generator for application-assigned identifiers (doesn't really generate anything).
 *
 * @since   2.0
 * @author  Benjamin Eberlei <kontakt@beberlei.de>
 * @author  Guilherme Blanco <guilhermeblanco@hotmail.com>
 * @author  Jonathan Wage <jonwage@gmail.com>
 * @author  Roman Borschel <roman@code-factory.org>
 */
class AssignedGenerator extends AbstractIdGenerator
{
    /**
     * Returns the identifier assigned to the given entity.
     *
     * @param EntityManager $em
     * @param object        $entity
     *
     * @return mixed
     *
     * @throws \Doctrine\ORM\ORMException
     *
     * @override
     */
    public function generate(EntityManager $em, $entity)
    {
        $class      = $em->getClassMetadata(get_class($entity));
        $idFields   = $class->getIdentifierFieldNames();
        $identifier = array();

        foreach ($idFields as $idField) {
            $value = $class->getFieldValue($entity, $idField);

            if ( ! isset($value)) {
                throw ORMException::entityMissingAssignedIdForField($entity, $idField);
            }

            if (isset($class->associationMappings[$idField])) {
                if ( ! $em->getUnitOfWork()->isInIdentityMap($value)) {
                    throw ORMException::entityMissingForeignAssignedId($entity, $value);
                }

                // NOTE: Single Columns as associated identifiers only allowed - this constraint it is enforced.
                $value = current($em->getUnitOfWork()->getEntityIdentifier($value));
            }

            $identifier[$idField] = $value;
        }

        return $identifier;
    }
}
