# Authors:
#     Endi S. Dewata <edewata@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
# Copyright (C) 2013 Red Hat, Inc.
# All rights reserved.
#

from __future__ import absolute_import
import os

import pki.server.upgrade


class MoveWebApplicationContextFile(
        pki.server.upgrade.PKIServerUpgradeScriptlet):

    def __init__(self):
        super(MoveWebApplicationContextFile, self).__init__()
        self.message = 'Move web application context file'

    def upgrade_instance(self, instance):

        self.upgrade_webapp(instance, 'ROOT')
        self.upgrade_webapp(instance, 'pki')

    def upgrade_subsystem(self, instance, subsystem):

        self.upgrade_webapp(instance, subsystem.name)

    def upgrade_webapp(self, instance, webapp):

        metainf_dir = os.path.join(
            instance.base_dir,
            'webapps',
            webapp,
            'META-INF')
        if not os.path.exists(metainf_dir):
            # upgrade already done
            return
        self.backup(metainf_dir)

        old_context_file = os.path.join(metainf_dir, 'context.xml')
        if not os.path.exists(old_context_file):
            # upgrade already done
            return
        self.backup(old_context_file)

        catalina_dir = os.path.join(instance.base_dir, 'conf', 'Catalina')
        self.backup(catalina_dir)

        localhost_dir = os.path.join(catalina_dir, 'localhost')
        self.backup(localhost_dir)

        new_context_file = os.path.join(localhost_dir, webapp + '.xml')
        self.backup(new_context_file)

        # prepare target folder
        if not os.path.exists(localhost_dir):
            os.makedirs(localhost_dir)

        # copy context file, don't overwrite existing file
        pki.util.copyfile(old_context_file, new_context_file, force=False)

        # set file and folder ownership
        pki.util.chown(catalina_dir, instance.uid, instance.gid)

        # remove old context file
        if os.path.exists(old_context_file):
            os.remove(old_context_file)

        # remove empty META-INF
        if not os.listdir(metainf_dir):
            os.rmdir(metainf_dir)
