/* $Id: FunctionDeclaration.hpp 5084 2010-12-17 16:24:58Z potyra $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __FUNCTION_DECLARATION_HPP_INCLUDED
#define __FUNCTION_DECLARATION_HPP_INCLUDED

#include "frontend/ast/Callable.hpp"
#include "frontend/ast/SubtypeIndication.hpp"
#include "frontend/misc/BuiltinFunction.hpp"
#include "frontend/visitor/GCBuiltins.hpp"

namespace ast {

/** declaration of a VHDL function */
class FunctionDeclaration : public Callable {
public:
	//! c'tor
	/** @param declName name of the function.
          * @param args interface (arguments the function takes).
	  * @param body corresponding SubprogBody, if any.
	  * @param returnT return type identifier of the function.
          * @param pureFunc is this function declared as pure?
	  * @param loc location of the FunctionDeclaration.
          */
	FunctionDeclaration(
		std::string *declName,
		std::list<ValDeclaration*> *args,
		SubtypeIndication *returnT,
		bool pureFunc,
		Location loc
		) : 	Callable(declName, args, 
				SubprogBody::PROG_KIND_FUNCTION, loc),
			returnType(returnT),
			builtin(NULL),
			gcBuiltin(NULL),
			isPure(pureFunc),
			isBuiltin(false) {}

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor &visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		assert(this->name);
		stream << "FUNCTION " << *this->name << '(';
		util::MiscUtil::listPut(this->arguments, stream, ", ");
		stream << ") RETURN " << this->returnType << ';';
	}

	/** return the name, that the intermediate code should use.
	 *  Overloaded from SymbolDeclaration, since the path name 
	 *  wouldn't be right for builtins.
	 */
	virtual std::string getICName(void) const {
		if (this->isBuiltin) {
			assert(this->name != NULL);
			return *this->name;
		}

		// not a bultin, reuse SymbolDeclaration's impl.
		return SymbolDeclaration::getICName();
	}

	
	/** return type of the function */
	SubtypeIndication *returnType;

	/** builtin function for constant propagation or NULL if not a 
	 *  builtin. */
	BuiltinFunction *builtin;

	/** inline intermediate code generation class or NULL if not a 
	 *  builtin */
	GCBuiltins *gcBuiltin;

	/** is the function declared as pure */
	bool isPure;

	/** is this a builtin function? */
	bool isBuiltin;

protected:
	/** Destructor */
	virtual ~FunctionDeclaration() {
		util::MiscUtil::terminate(this->returnType);

		delete this->builtin;
		this->builtin = NULL;

		delete this->gcBuiltin;
		this->gcBuiltin = NULL;
	}
};

};

#endif /* __FUNCTION_DECLARATION_HPP_INCLUDED */
