#include "glotski.h"
#include <stdlib.h>
#include <stdio.h>

level *makelevel(int xsize, int ysize) { /* Level constructor */
  level *newlevel = (level *)calloc(1, sizeof(level));
  movelist *newmove = (movelist *)calloc(1, sizeof(movelist));
  glot **newfield = (glot **)calloc(xsize*ysize,sizeof(glot *));
  if (!newlevel || !newfield) {
    fprintf(stderr, "glotski: Needed %d bytes to allocate level\n",
	    sizeof(level) + xsize*ysize*sizeof(glot *) + 62*sizeof(int) +
	    sizeof(movelist));
    exit(1);
  }
  /* state, goal, numglots, nextid, equiv automatically set to 0 by calloc */
  newlevel->usize[X] = newlevel->usize[Y] = 32.0;
  newlevel->scale[X] = newlevel->scale[Y] = 
    newlevel->zoom[X] = newlevel->zoom[Y] = 1.0;
  newlevel->gzoom[X] = newlevel->gzoom[Y] = 0.5;
  newlevel->field = newfield;
  newlevel->size[0] = xsize;
  newlevel->size[1] = ysize;
  newlevel->uniteid = newlevel->showgoal = 1;
  newlevel->move = newmove; /* Placeholder to allow undo of first move */
  return newlevel;
}

glot *makeglot(level *mylevel, int xloc, int yloc) { /* Glot constructor */
  setuplist *newnode = (setuplist *)calloc(1, sizeof(setuplist));
  glot *newglot = (glot *)calloc(1, sizeof(glot));
  if (!newglot || !newnode) {
    fprintf(stderr, "glotski: Needed %d bytes to allocate glot\n",
	    sizeof(glot) + sizeof(setuplist));
    exit(1);
  }

  newnode->next = mylevel->setup; /* Prepend the glot */
  mylevel->setup = newnode;
  newnode->this = newglot;
  
  /* field, label start as 0 */ 
  newglot->type = BIMOBILE;
  newglot->red = 0; newglot->green = 127; newglot->blue = 0;
  newglot->loc[X] = xloc; newglot->loc[Y] = yloc;
  newglot->size[X] = newglot->size[Y] = 1;
  newglot->parent = mylevel;
  newglot->id = mylevel->nextid++;
  return newglot;
}

goal *makegoal(level *mylevel) { /* Goal constructor */
  goal *newgoal = calloc(1, sizeof(goal));
  glot **newfield = (glot **)calloc(mylevel->size[X]*mylevel->size[Y], 
				    sizeof(glot *));
  if (!newgoal || !newfield) {
    fprintf(stderr, "glotski: Needed %d bytes to allocate goal\n",
	    mylevel->size[X]*mylevel->size[Y]*sizeof(glot *) + sizeof(goal));
    exit(1);    
  }

  newgoal->next = mylevel->mygoal;    /* Prepend goal: Last goal first. */
  if (mylevel->mygoal) mylevel->mygoal->prev = newgoal;
  mylevel->mygoal = newgoal;

  newgoal->field = newfield;
  newgoal->size[X] = mylevel->size[X];
  newgoal->size[Y] = mylevel->size[Y];
  mylevel->numgoals++;

  return newgoal;
}

movelist *makemove(level *mylevel) {  /* Move constructor */
  movelist *newmove;
  if (mylevel->move && mylevel->move->next) return NULL;
  newmove = (movelist *)calloc(1, sizeof(movelist));
  newmove->prev = mylevel->move;
  if (mylevel->move) mylevel->move->next = newmove;
  mylevel->move = newmove;
  return newmove;
}

void deleteglot(glot *myglot) { /* Glot destructor */
  if (!myglot) return;
  if (myglot->field) free(myglot->field);
  /* if (myglot->label) free(myglot->label); */
  free(myglot);
}

void deletegoal(goal *mygoal) { /* Goal destructor */
  if (!mygoal) return;
  if (mygoal->next) deletegoal(mygoal->next); /* Recursive, how special */
  free(mygoal->field);
  free(mygoal);
}

void popgoal(goal **mygoal) { /* Bring next goal to front */
  goal *nextgoal;
  if (!mygoal || !(*mygoal)) return;
  nextgoal = (*mygoal)->next;
  (*mygoal)->next = NULL;
  deletegoal(*mygoal);
  *mygoal = nextgoal;
}

void deletesetup(setuplist *mysetup) { /* List destructor */
  if (!mysetup) return;
  if (mysetup->next) deletesetup(mysetup->next);
  deleteglot(mysetup->this);
  free(mysetup);
}

void deletemove(movelist *mymove) { /* Move destructor */
  if (!mymove) return;
  if (mymove->next) deletemove(mymove->next);
  free(mymove);
}

void popmove(movelist **mymove) {
  movelist *prevmove;
  if (!mymove || !(*mymove)) return;
  prevmove = (*mymove)->prev;
  if (prevmove) prevmove->next = (*mymove)->next;
  free(*mymove);
  *mymove = prevmove;
}

void deletelevel(level *mylevel) { /* Free a level and all its contents */
  if (!mylevel) return;
  if (mylevel->field) free(mylevel->field);
  if (mylevel->setup) deletesetup(mylevel->setup);
  if (mylevel->mygoal) deletegoal(mylevel->mygoal);
  if (mylevel->equiv) free(mylevel->equiv);
  free(mylevel);
}

int goalequiv(level *mylevel) { /* See if a level is equivalent to its goal */
  int x, y;
  int perfect = 1;
  goal *mygoal = mylevel->mygoal;
  glot *tmp1, *tmp2;
  if (!mygoal) return 0;
  for (x = 0; x < mylevel->size[X]; x++) {
    for (y = 0; y < mylevel->size[Y]; y++) {
      tmp1 = LOOKUP(mygoal, x, y); tmp2 = LOOKUP(mylevel, x, y);
      if ((!tmp2 && tmp1) || 
	  (tmp1 && tmp2 && !eq_equiv(mylevel->equiv, tmp1->id, tmp2->id))) {
	perfect = 0;
	break;
      }      
    }
    if (perfect == 0) break;
  }
  return perfect;
}

int glotplace(glot *myglot, int xloc, int yloc, int perform) {
  int x, y;
  int perfect = 1;
  if (xloc < 0 || yloc < 0 || 
      xloc + myglot->size[X] > myglot->parent->size[X] ||
      yloc + myglot->size[Y] > myglot->parent->size[Y]) return 0;
  /* First, remove glot from current spot */
  for (x = 0; x < myglot->size[X]; x++) 
    for (y = 0; y < myglot->size[Y]; y++)
      if (LOOKUP(myglot, x, y))
	LOOKUP(myglot->parent, 
	       x + myglot->loc[X], y + myglot->loc[Y]) = NULL;

  /* Then, see if glot can be placed on new spot (known in bounds) */
  for (x = 0; x < myglot->size[X]; x++) {
    for (y = 0; y < myglot->size[Y]; y++)
      if (LOOKUP(myglot, x, y))
	if (LOOKUP(myglot->parent, x + xloc, y + yloc) != NULL) {
	  perfect = 0;
	  break;
	}
    if (perfect == 0) break;
  }

  /* Now, replace it and quit if you're not supposed to perform */
  if (perfect == 0 || perform == 0) { 
    if (!glotplace(myglot, myglot->loc[X], myglot->loc[Y], 1)) {
      fprintf(stderr, "glotski: Unable to replace glot\n");
      exit(1);
    }
    return perfect;
  }

  /* Finally, place the glot knowing it can be placed */
  for (x = 0; x < myglot->size[X]; x++) 
    for (y = 0; y < myglot->size[Y]; y++)
      if (LOOKUP(myglot, x, y))
	LOOKUP(myglot->parent, x + xloc, y + yloc) = myglot;

  myglot->loc[X] = xloc; myglot->loc[Y] = yloc;

  return 1;
}

int glotmove(glot *moving, dirtype dir, int perform) { /* Move glot some dir */
  int xloc, yloc;

  if (!COMPAT(moving, dir)) return 0;

  if (AXIS(dir) == X) {
    xloc = moving->loc[X] + 1 - 2*!SIGN(dir);
    yloc = moving->loc[Y];
  } else {
    xloc = moving->loc[X];
    yloc = moving->loc[Y] + 1 - 2*!SIGN(dir);
  }

  return glotplace(moving, xloc, yloc, perform);
}

void undo(level *mylevel) {
  if (mylevel->move && mylevel->move->prev) {
    moveelement(mylevel->move->myglot, mylevel->move->startloc[X],
		mylevel->move->startloc[Y]);
    if (mylevel->move->goalsat) {
      if (mylevel->state != mylevel->numgoals)
	mylevel->mygoal = mylevel->mygoal->prev; /* Last goal never reached */
      mylevel->state--;
      redrawgoal(mylevel);
    }
    mylevel->move = mylevel->move->prev;
    mylevel->nummoves--;
    updatestatus(mylevel);
  }
}

void redo(level *mylevel) {
  if (mylevel->move && mylevel->move->next) {
    mylevel->move = mylevel->move->next;
    moveelement(mylevel->move->myglot, mylevel->move->endloc[X],
		mylevel->move->endloc[Y]);
    if (mylevel->move->goalsat) {
      mylevel->mygoal = mylevel->mygoal->next;
      mylevel->state++;
      redrawgoal(mylevel);
    }
    mylevel->nummoves++;
    updatestatus(mylevel);
  }
}



