//*****************************************************************************
//                                  SpinCtrl.hpp                              *
//                                 --------------                             *
// Description : This class extends wxPanel, and is a replacement for the     *
//               wxWidgets library class wxSpinCtrl. It adds some useful      *
//               functionality :                                              *
//                - floating point or integer numbers can be displayed,       *
//                - incrementing by orders of magnitude not just a constant,  *
//                - variable step size incrementing.                          *
// Started     : 20/03/2004                                                   *
// Last Update : 11/05/2011                                                   *
// Copyright   : (C) 2004 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#ifndef SPINCTRL_HPP
#define SPINCTRL_HPP

// System Includes

#include <iostream>
#include <cmath>

// wxWidgets Includes

#include <wx/wx.h>
#include <wx/spinbutt.h>

// Application Includes

#include "TypeDefs.hpp"
#include "ConvertType.hpp"

// Local Constant Declarations

#define  SPN_MAXLEN    9  // Maximum chars. that may be entered into text ctrl.
#define  SPN_PERIOD   80  // The spin button repeat period (in msec)

//*****************************************************************************

class SpinCtrl : public wxPanel
{
  private :

    // Display objects
    wxTextCtrl    m_oTxtValue;
    wxSpinButton  m_oSbnValue;

    eVarType      m_eVarType;   // The variable type to be displayed
    float         m_fDefValue;  // Default value
    float         m_fMinValue;  // Minimum value
    float         m_fMaxValue;  // Maximum value
    float         m_fMinIncSz;  // Minimum increment size
    float         m_fMaxIncSz;  // Maximum increment size

    void  DoLayout( void );

  public :

    SpinCtrl( eVarType eVType = eVAR_FLT );
   ~SpinCtrl( );

    bool  bCreate   ( wxWindow * poWin, wxWindowID oWinID, int iWidth=-1 );
    bool  bIsCreated( void ) { return( GetParent()!=NULL ? TRUE : FALSE ); }

    bool  bClear( void );
    bool  bSetDefaults( void );

    bool  bSetVarType ( eVarType eNewVarType );
    bool  bSetValue   ( int iValue ) { return( bSetValue( (float) iValue ) ); }
    bool  bSetValue   ( float fValue );
    bool  bSetValue   ( const wxString & rosValue );
    bool  bSetDefValue( float fDefValue );
    bool  bSetMinValue( float fMinValue )
                             { return( bSetRange( fMinValue, m_fMaxValue ) ); }
    bool  bSetMaxValue( float fMaxValue )
                             { return( bSetRange( m_fMinValue, fMaxValue ) ); }
    bool  bSetIncSz   ( float fMinIncSz, float fMaxIncSz=-1.0 );
    bool  bSetRange   ( float fMinValue, float fMaxValue );
    bool  bSetParms   ( float fDefValue, float fMinValue, float fMaxValue,
                        float fMinIncSz=-1.0, float fMaxIncSz=-1.0 );

          eVarType     eGetVType   ( void ) { return( m_eVarType ); }
          int          iGetValue   ( void );
          float        fGetValue   ( void );
    const wxString & rosGetValue   ( void );
          float        fGetDefValue( void ) { return( m_fDefValue ); }
          float        fGetMinValue( void ) { return( m_fMinValue ); }
          float        fGetMaxValue( void ) { return( m_fMaxValue ); }
          float        fGetMinIncSz( void ) { return( m_fMinIncSz ); }
          float        fGetMaxIncSz( void ) { return( m_fMaxIncSz ); }

    // Event handlers
    void  OnTxtChar  ( wxKeyEvent     & roEvtKey );
    void  OnTxtMaxLen( wxCommandEvent & roEvtCmd );
    void  OnSbnInc   ( wxSpinEvent    & roEvtSpn );
    void  OnSbnDec   ( wxSpinEvent    & roEvtSpn );

    // In order to be able to react to a menu command, it must be given a
    // unique identifier such as a const or an enum.
    enum ePnlItemID
    {
      ID_TXTCTRL = 1,
      ID_SPINBTN,

      ID_UNUSED,

      ID_FST = ID_TXTCTRL,
      ID_LST = ID_SPINBTN
    };

    // Leave this as the last line as private access is envoked by macro
    DECLARE_EVENT_TABLE( )
};

//*****************************************************************************

#endif // SPINCTRL_HPP
