/* GStreamer unix file-descriptor source/sink tests
 *
 * Copyright (C) 2023 Netflix Inc.
 *  Author: Xavier Claessens <xavier.claessens@collabora.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gst/check/gstcheck.h>
#include <glib/gstdio.h>

static void
wait_preroll (GstElement * element)
{
  GstStateChangeReturn state_res =
      gst_element_set_state (element, GST_STATE_PLAYING);
  fail_unless (state_res != GST_STATE_CHANGE_FAILURE);
  state_res = gst_element_get_state (element, NULL, NULL, GST_CLOCK_TIME_NONE);
  fail_unless (state_res == GST_STATE_CHANGE_SUCCESS);
}

static GstPadProbeReturn
buffer_pad_probe_cb (GstPad * pad, GstPadProbeInfo * info, gpointer user_data)
{
  GstBuffer *buffer = GST_PAD_PROBE_INFO_BUFFER (info);
  if (buffer != NULL) {
    GstCustomMeta *cmeta =
        gst_buffer_add_custom_meta (buffer, "unix-fd-custom-meta");
    GstStructure *s = gst_custom_meta_get_structure (cmeta);
    gst_structure_set (s, "field", G_TYPE_INT, 42, NULL);
  }
  return GST_PAD_PROBE_OK;
}

GST_START_TEST (test_unixfd_videotestsrc)
{
  GError *error = NULL;

  const gchar *tags[] = { NULL };
  gst_meta_register_custom ("unix-fd-custom-meta", tags, NULL, NULL, NULL);

  /* Ensure we don't have socket from previous failed test */
  gchar *socket_path =
      g_strdup_printf ("%s/unixfd-test-socket", g_get_user_runtime_dir ());
  if (g_file_test (socket_path, G_FILE_TEST_EXISTS)) {
    g_unlink (socket_path);
  }

  /* Setup source */
  gchar *pipeline_str =
      g_strdup_printf ("videotestsrc name=src ! unixfdsink socket-path=%s",
      socket_path);
  GstElement *pipeline_service = gst_parse_launch (pipeline_str, &error);
  g_assert_no_error (error);
  g_free (pipeline_str);

  /* Add a custom meta on each buffer */
  GstElement *src = gst_bin_get_by_name (GST_BIN (pipeline_service), "src");
  GstPad *pad = gst_element_get_static_pad (src, "src");
  gst_pad_add_probe (pad, GST_PAD_PROBE_TYPE_BUFFER, buffer_pad_probe_cb, NULL,
      NULL);
  gst_object_unref (src);
  gst_object_unref (pad);

  wait_preroll (pipeline_service);

  /* Setup sink */
  pipeline_str =
      g_strdup_printf ("unixfdsrc socket-path=%s ! fakesink name=sink",
      socket_path);
  GstElement *pipeline_client_1 = gst_parse_launch (pipeline_str, &error);
  g_assert_no_error (error);
  wait_preroll (pipeline_client_1);

  /* disconnect, reconnect */
  fail_unless (gst_element_set_state (pipeline_client_1,
          GST_STATE_READY) == GST_STATE_CHANGE_SUCCESS);
  wait_preroll (pipeline_client_1);

  /* Connect 2nd sink */
  GstElement *pipeline_client_2 = gst_parse_launch (pipeline_str, &error);
  g_assert_no_error (error);
  wait_preroll (pipeline_client_2);

  /* Check we received our custom meta */
  GstSample *sample;
  GstElement *sink = gst_bin_get_by_name (GST_BIN (pipeline_client_2), "sink");
  g_object_get (sink, "last-sample", &sample, NULL);
  fail_unless (sample);
  GstBuffer *buffer = gst_sample_get_buffer (sample);
  GstCustomMeta *cmeta =
      gst_buffer_get_custom_meta (buffer, "unix-fd-custom-meta");
  fail_unless (cmeta);
  GstStructure *s = gst_custom_meta_get_structure (cmeta);
  gint value;
  fail_unless (gst_structure_get_int (s, "field", &value));
  fail_unless_equals_int (value, 42);
  gst_object_unref (sink);
  gst_sample_unref (sample);

  /* Teardown */
  fail_unless (gst_element_set_state (pipeline_client_1,
          GST_STATE_NULL) == GST_STATE_CHANGE_SUCCESS);
  fail_unless (gst_element_set_state (pipeline_client_2,
          GST_STATE_NULL) == GST_STATE_CHANGE_SUCCESS);
  fail_unless (gst_element_set_state (pipeline_service,
          GST_STATE_NULL) == GST_STATE_CHANGE_SUCCESS);
  fail_if (g_file_test (socket_path, G_FILE_TEST_EXISTS));

  gst_object_unref (pipeline_service);
  gst_object_unref (pipeline_client_1);
  gst_object_unref (pipeline_client_2);
  g_free (socket_path);
  g_free (pipeline_str);
}

GST_END_TEST;

static Suite *
unixfd_suite (void)
{
  Suite *s = suite_create ("unixfd");
  TCase *tc = tcase_create ("unixfd");

  suite_add_tcase (s, tc);
  tcase_add_test (tc, test_unixfd_videotestsrc);

  return s;
}

GST_CHECK_MAIN (unixfd);
