/*
 * jclassinfo
 * Copyright (C) 2003  Nicos Panayides
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Nicos Panayides
 * anarxia@gmx.net
 *
 * $Id: field.c,v 1.9 2004/05/03 08:45:51 anarxia Exp $
 */

#include "config.h"
#include <stdlib.h>
#include <string.h>
#include <jclass/jclass.h>
#include "common.h"
#include "attributes.h"
#include "attributes_xml.h"
#include "field.h"

void field_print(Field* field, ConstantPool* cpool, int do_flag)
{
	int i;
	IntType int_type;
	int code_index = -1;
	int constant_value_index = -1;
	int deprecated_index = -1;
	int exceptions_index = -1;
	int synthetic_index = -1;
	char* descriptor;
	char* access_flag;
	char* field_name;
	char* type;
	char* params;
	
	int_type = INT_IS_INT;
	
	field_name = jclass_field_get_name(field, cpool);
	descriptor = jclass_field_get_descriptor(field, cpool);

	access_flag = jclass_access_flag_to_string(field->access_flags, 0);
	if(access_flag[0] != '\0')
		printf("%s ",access_flag);
		
	free(access_flag);
	
	/* special case for constructor */
	if(!strcmp(field_name,"<init>"))
	{
		free(field_name);
		field_name = jclass_cp_get_class_name(cpool, cpool->this_class, 0);
		printf("%s", field_name);
	}
	else if(!strcmp(field_name,"<clinit>"))
	{
		if (!(do_flag & DISASM))
			printf("{}");
	}
	else
	{
		type = jclass_descriptor_get_type(descriptor);
		if(!strcmp(type,"boolean"))
			int_type = INT_IS_BOOLEAN;
		if(!strcmp(type,"char"))
			int_type = INT_IS_CHAR;		
		
		printf("%s %s", type, field_name);
		free(type);
	}
	
	/* special case for static initializer */
	if(strcmp(field_name,"<clinit>"))
	{
		params = jclass_descriptor_get_parameters(descriptor);
		
		printf("%s ", params);
		free(params);
	}
	
	free(descriptor);
	free(field_name);
	
	for(i=0; i < field->attributes_count; i++)
	{
		if(jclass_attribute_container_has_attribute(&field->attributes[i], "Code", cpool))
			code_index = i;
		else if (jclass_attribute_container_has_attribute(&field->attributes[i], "Deprecated", cpool))
			deprecated_index = i;
		else if (jclass_attribute_container_has_attribute(&field->attributes[i], "Exceptions", cpool))
			exceptions_index = i;
		else if (jclass_attribute_container_has_attribute(&field->attributes[i], "ConstantValue", cpool))
			constant_value_index = i;
		else if (jclass_attribute_container_has_attribute(&field->attributes[i], "Synthetic", cpool))
			synthetic_index = i;
	}

	if (constant_value_index >= 0)
		attribute_container_print(&field->attributes[constant_value_index], 
			int_type, do_flag, cpool);

	if (exceptions_index >=0)
		attribute_container_print(&field->attributes[exceptions_index], 
			int_type, do_flag, cpool);

	if (deprecated_index >=0 && (do_flag & ATTRIBUTES))
		attribute_container_print(&field->attributes[deprecated_index], 
			int_type, do_flag, cpool);

	if (synthetic_index >=0 && (do_flag & ATTRIBUTES))
		attribute_container_print(&field->attributes[synthetic_index], 
			int_type, do_flag, cpool);

	for(i=0; i < field->attributes_count; i++)
	{
		if((do_flag & ATTRIBUTES) && 
			i != constant_value_index && i != exceptions_index && i != deprecated_index
			&& i != synthetic_index && i != code_index)
		{
			attribute_container_print(&field->attributes[i], int_type, 0, cpool);
		}
	}
	if(code_index >= 0)
		attribute_container_print(&field->attributes[code_index], 
			int_type, do_flag, cpool);		
	
	puts("");
}

void field_print_xml(Field* field, ConstantPool* cpool, int do_flag, int is_method)
{
	int i;
	IntType int_type;
	int code_index = -1;
	char* descriptor;
	char* access_flag;
	char* field_name;
	char* type;
	char** params;
	
	int_type = INT_IS_INT;
	
	if(is_method)
		printf("<method");
	else
		printf("<field");
	
	field_name = jclass_field_get_name(field, cpool);
	descriptor = jclass_field_get_descriptor(field, cpool);

	access_flag = jclass_access_flag_to_string(field->access_flags, 0);
	if(access_flag[0] != '\0')
		printf(" access=\"%s\"",access_flag);
		
	free(access_flag);
	
	/* special case for constructor */
	if(!strcmp(field_name,"<init>"))
	{
		free(field_name);
		field_name = jclass_cp_get_class_name(cpool, cpool->this_class, 0);
		printf(" name=\"%s\"", field_name);
	}
	else if(!strcmp(field_name,"<clinit>"))
	{
		printf(" name=\"static\"");
	}
	else
	{
		type = jclass_descriptor_get_type(descriptor);
		if(!strcmp(type,"boolean"))
			int_type = INT_IS_BOOLEAN;
		if(!strcmp(type,"char"))
			int_type = INT_IS_CHAR;		
		
		printf(" type=\"%s\" name=\"%s\"", type, field_name);
		free(type);
	}
	puts(">");

	/* special case for static initializer */
	if(strcmp(field_name,"<clinit>") && is_method)
	{
		params = jclass_descriptor_get_parameters_array(descriptor);
		i=0;
		while (params[i] != NULL) {
			printf("<param index=\"%d\" type=\"%s\"/>\n", i, params[i]);
			free(params[i]);
			i++;
		}
		free(params);
	}
	
	
	free(descriptor);
	free(field_name);
	
	for(i=0; i < field->attributes_count; i++)
	{
		if(jclass_attribute_container_has_attribute(&field->attributes[i], "Code", cpool))
			code_index = i;
		else if (jclass_attribute_container_has_attribute(&field->attributes[i], "Exceptions", cpool) ||
				 jclass_attribute_container_has_attribute(&field->attributes[i], "ConstantValue", cpool))
			attribute_container_print_xml(&field->attributes[i], 
				int_type, 0, cpool);	
	}

	for(i=0; i < field->attributes_count; i++)
	{
		if((do_flag & ATTRIBUTES) &&
				!jclass_attribute_container_has_attribute(&field->attributes[i], "ConstantValue", cpool) &&
				!jclass_attribute_container_has_attribute(&field->attributes[i], "Exceptions", cpool))
		{
			attribute_container_print_xml(&field->attributes[i], 
				int_type, 0, cpool);
		}
	}

	if(code_index >= 0)
		attribute_container_print_xml(&field->attributes[code_index], 
			int_type, do_flag, cpool);		
	
	if(is_method)
		puts("</method>");
	else
		puts("</field>");
}
