/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */



#include <omp.h>
#include <limits>


/////////////////////// Qt includes
#include <QDebug>
#include <QFile>
#include <qmath.h>


/////////////////////// Local includes
#include <libmass/MassSpectrum.hpp>
#include <globals/globals.hpp>

#include <pwiz/data/msdata/MSDataFile.hpp>
#include <pwiz/data/msdata/DefaultReaderList.hpp>

using namespace pwiz::msdata;


namespace msXpSlibmass
{

	double MassSpectrum::m_maxAvgMzBinSize = 0.05;

	double MassSpectrum::m_maxBinSizeStdDev = MassSpectrum::m_maxAvgMzBinSize / 5;

	int massSpectrumMetaTypeId = qRegisterMetaType<msXpSlibmass::MassSpectrum>("msXpSlibmass::MasSpectrum");

	//! Construct a totally empty MassSpectrum
	MassSpectrum::MassSpectrum()
	{
		//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()"
		//<< "default constructor"
		//<< this;
	}


	//! Copy constructor (with deep copy of the Trace's DataPoint objects)
	MassSpectrum::MassSpectrum(const MassSpectrum &other)
	{
		//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()"
		//<< "copy constructor, other :" << &other << "this:" << this;

		m_title = other.m_title;
		m_lastModifIdx = other.m_lastModifIdx;
		m_lastBinIdx = other.m_lastBinIdx;
		m_rt = other.m_rt;
		m_dt = other.m_dt;
		m_decimalPlaces = other.m_decimalPlaces;
		m_greatestStep = other.m_greatestStep;
		m_binCount = other.m_binCount;
		m_isBinnable = other.m_isBinnable;
		m_binMinMz = other.m_binMinMz;
		m_binMaxMz = other.m_binMaxMz;
		m_mzShift = other.m_mzShift;

		for(int iter = 0; iter < other.size(); ++iter)
			append(new DataPoint(*(other.at(iter))));
	}


	//! Construct a MassSpectrum with \p title
	/*!
	*/
	MassSpectrum::MassSpectrum(const QString &title):
		Trace{title}
	{
	}


	//! Destruct the MassSpectrum instance.
	MassSpectrum::~MassSpectrum()
	{
		//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()";
	}


	//! Assignment operator (with deep copy of \p other::Trace's DataPoint objects into \c this.
	/*!

		This function starts by deleting all of \c this instance DataPoint objects.
		Then all of the DataPoint instances in \p other are duplicated into \e new
		DataPoint instances appended to \c this MassSpectrum object.

		The first use of this assignment operator was for JavaScripting because
		translation between JS object and C++ object required deep copy of all the
		data members of MassSpectrum.

*/
	MassSpectrum &
		MassSpectrum::operator=(const MassSpectrum &other)
		{
			if(this == &other)
				return *this;

			// Make sure we start by deleting all of our peaks.
			Trace::deleteDataPoints();

			m_title = other.m_title;
			m_lastModifIdx = other.m_lastModifIdx;
			m_lastBinIdx = other.m_lastBinIdx;
			m_rt = other.m_rt;
			m_dt = other.m_dt;
			m_decimalPlaces = other.m_decimalPlaces;
			m_greatestStep = other.m_greatestStep;
			m_binCount = other.m_binCount;
			m_isBinnable = other.m_isBinnable;
			m_binMinMz = other.m_binMinMz;
			m_binMaxMz = other.m_binMaxMz;
			m_mzShift = other.m_mzShift;

			// And now create as many *new* peaks as there are in other.
			for(int iter = 0; iter < other.size(); ++iter)
				append(new DataPoint(*(other.at(iter))));

			return *this;
		}


	//! Reset the MassSpectrum
	void
		MassSpectrum::reset()
		{
			Trace::reset();

			m_lastBinIdx = 0;
			m_binCount = 0;

			m_dt = 0;
			m_rt = 0;

			m_greatestStep = qSNaN();

			m_isBinnable = false;

			m_binMinMz = 0;
			m_binMaxMz = 0;

			m_mzShift = 0;
		}


	//! Get the retention time member datum value
	/*!
		\return the retention time value at which \c this spectrum was acquired.
		*/
	double
		MassSpectrum::rt() const
		{
			return m_rt;
		}


	//! Get a reference to the retention time member datum value
	/*!
		\return a reference to the retention time value at which \c this spectrum
		was acquired. The reference handle allows for modification of the retention
		time value.
		*/
	double &
		MassSpectrum::rrt()
		{
			return m_rt;
		}


	//! Get the drift time member datum value
	double
		MassSpectrum::dt() const
		{
			return m_dt;
		}


	//! Get a reference to the drift time member datum value
	double &
		MassSpectrum::rdt()
		{
			return m_dt;
		}


	//! Initialize the MassSpectrum with data
	/*!

		Calls Trace::initialize() and sets m_rt.

		Note that mzList and iList cannot have different sizes otherwise the program
		crashes.

		\param mzList list of m/z values
		\param iList list of intensity values
		\param rt the retention time at which the mass spectrum was acquired

*/
	int
		MassSpectrum::initializeRt(const QList<double> &mzList,
				const QList<double> &iList, double rt)
		{
			Trace::initialize(mzList, iList);
			m_rt = rt;

			return size();
		}


	//! Initialize the MassSpectrum with data
	/*!

		Calls Trace::initialize() and sets m_dt.

		Note that mzList and iList cannot have different sizes otherwise the program
		crashes.

		\param mzList list of m/z values
		\param iList list of intensity values
		\param dt the drift time at which the mass spectrum was acquired
		*/
	int
		MassSpectrum::initializeDt(const QList<double> &mzList,
				const QList<double> &iList, double dt)
		{
			Trace::initialize(mzList, iList);

			m_dt = dt;

			return size();
		}


	//! Initialize the MassSpectrum with data
	/*!

		Overload of MassSpectrum::initialize().

*/
	int	
		MassSpectrum::initializeRt(const QList<double> &mzList,
				const QList<double> &iList, double rt,
				double mzStart, double mzEnd)
		{
			Trace::initialize(mzList, iList, mzStart, mzEnd);

			m_rt = rt;

			return size();
		}

	//! Initialize the MassSpectrum with data
	/*!

		Overload of MassSpectrum::initialize().
		*/
	int
		MassSpectrum::initializeDt(const QList<double> &mzList,
				const QList<double> &iList, double dt,
				double mzStart, double mzEnd)
		{
			Trace::initialize(mzList, iList, mzStart, mzEnd);

			m_dt = dt;

			return size();
		}


	//! Initialize the MassSpectrum with data
	/*!

		Overload of MassSpectrum::initialize().
		*/
	int
		MassSpectrum::initialize(const QByteArray *mzByteArray,
				const QByteArray *iByteArray, int compressionType, 
				double mzStart, double mzEnd)
		{
			Trace::initialize(mzByteArray, iByteArray, compressionType, mzStart, mzEnd);

			return size();
		}


	//! Initialize the MassSpectrum with data
	/*!

		Overload of MassSpectrum::initialize().
		*/
	int 
		MassSpectrum::initialize(const Trace &trace)
		{
			return Trace::initialize(trace);
		}


	//! Setup the binning infrastructure
	/*!

		Binning is an operation by which a set of mass spectra (at least two
		spectra) are analyzed and checked for a number of properties. If these
		properties conform to specific criteria, the binning is setup in such a
		manner that any subsequent operation (mostly combine operations) are taking
		advantage of the binning that was setup. The binning infrastructure is only
		created if:

		- the spectra in the \p massSpectra QList<MassSpectrum *> have the same
		length.  This is logical: if we want to make a binning operation on mass
		spectra, we need to have spectra that have the same number of points because
		binning involves creating bins that faithfully represent the initial mass
		data. If the spectra do not have the same size and the binning is done on a
		shorter spectrum than others, then, when we will perform combinations there
		will be missing bins and thus a part of the data will be concentrated in the
		last bin!

		Ensuring that the spectra have the same size helps ensuring that the m/z
		intervals between points keeps being the same throughout the spectra in \p
		massSpectra, although it might be a non-constant interval throughout the m/z
		range, as in the case of TOF data.

		- \c this spectrum is empty (\c this spectrum is only a holder spectrum for
		later combinations to be performed according to the binning setup).

		Overall working of this function:

		If \p massSpectra is empty, return false.

		The mass spectra in \p massSpectra are checked to establish if they all have
		approximately the same length. If not, then m_isBinnable is set to false and
		the function returns true. The computation otherwise goes on if the
		following condition is true: \c this mass spectrum is empty. If \c this mass
		spectrum is not empty, the program crashes.

		Then the characteristics of the \p massSpectra list are gotten (minimum and
		maximum m/z values of the whole set of mass spectra.). Use the first
		spectrum of the \p massSpectra list to fill-in the bins, that is, fill-in a
		list of double values with all the steps in the spectrum's m/z data (an
		interval between a given m/z value and the following one is called a step).
		Get from fillInBins() the greatest m/z step in that spectrum. Check that the
		bin list contains binnable data. Binnable data are data that have uniform
		steps (relatively similar steps) and steps that are much less than unity. If
		the data are binnable, then setupBins() is called to terminate the binning
		setup. \c m_binCount is set to the number of bins in the bin list.

		\param massSpectra list of MassSpectrum instances

		\return true in all cases unless \p massSpectra is empty, in which case it
		returns false.

*/
	bool
		MassSpectrum::setupBinnability(const QList<MassSpectrum *> &massSpectra)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" ;

			// Since setting up the bins must be performed only once, and that
			// operation needs to be performed prior to the first combination, we only
			// perform binning if this spectrum is empty, that is, it is the pristine
			// recipient of a combination operation yet to be started.

			if(size())
				qFatal("Fatal error at %s@%d -- %s. "
						"Setting up bins means that *this spectrum is empty. Which it is not."
						" Programming error. Program aborted.",
						__FILE__, __LINE__, __FUNCTION__);

			// If there are no spectra in the list, then we have nothing to do.

			if(massSpectra.size() == 0)
			{
				return false;
			}

			bool areSameLengthSpectra = MassSpectrum::areSpectraSameLength(massSpectra);

			if(!areSameLengthSpectra)
			{
				m_isBinnable = false;
				return true;
			}

			//qDebug() << __FILE__ << __LINE__
			//<< "Spectrum list has same-length spectra ?" << areSameLengthSpectra
			//<< "and might be binnable";

			if(areSameLengthSpectra)
			{
				double minMz = 0;
				double maxMz = 0;

				QList<double> binList;

				MassSpectrum::mzMinMax(massSpectra, &minMz, &maxMz);

				m_binMinMz = minMz;
				m_binMaxMz = maxMz;

				// Use the first spectrum of the list to fill in the binList

				MassSpectrum *massSpectrum = static_cast<MassSpectrum *>(massSpectra.first());

				if(qIsNaN(massSpectrum->fillInBins(&binList)))
					return false;

				//qWarning() << __FILE__ << __LINE__
				//<< QString("First spectrum to combine has %1 peaks and produced %2 bins:")
				//.arg(massSpectra.first()->size())
				//.arg(binList.size());
				//<< binList;

				m_isBinnable = isBinnable(binList);

				//qDebug() << __FILE__ << __LINE__
				//<< "Spectrum is binnable:" << m_isBinnable;

				if(m_isBinnable)
				{

					// Now that we know the characteristics of the mass spec list, we can
					// configure *this spectrum if not done already, that is, if size() is 0.

					m_binCount = binList.size();

					//qDebug() << __FILE__ << __LINE__
					//<< "minMz:" << minMz
					//<< "maxMz:" << maxMz
					//<< "spectrum size:" << size()
					//<< "m_binCount:" << m_binCount;

					setupBins(binList);

					//qDebug() << __FILE__ << __LINE__
					//<< "After having setup the bins, the number of data points:"
					//<< size();
				}
			}

			return true;
		}


	//! Fill-in \p binList with all the m/z intervals in \c this spectrum
	/*!

		A bin is a m/z interval separating two consecutive m/z values in a mass
		spectrum. By definition, a mass spectrum that has n DataPoint instances has
		(n-1) intervals. A mass spectrum with less than two DataPoint instances is by
		definition not binnable.

		This function postulates that the data in the mass spectrum are sorted in
		key (that is, m/z) increasing order. If not, the program crashes.

		\c this list of DataPoint instances is iterated into and for each previous \e
		vs current m/z value of two contiguous DataPoint instances, the (current -
		previous) m/z difference is computed and appended in \p binList.

		\param binList pointer to a list of double values corresponding to all the
		m/z intervals in the spectrum. Cannot be Q_NULLPTR.

		\return a double value with the greatest step (interval) found in \c this
		mass spectrum.

*/
	double
		MassSpectrum::fillInBins(QList<double> *binList)
		{
			if(binList == Q_NULLPTR)
				qFatal("Fatal error at %s@%d. Program aborted.",
						__FILE__, __LINE__);

			// We need at least two data points in the spectrum to compute the step
			// between the second and the first.

			if(size() < 2)
			{
				return qSNaN();
			}

			double greatestStep = 0;
			double tempStep = 0;

			double prevMz = at(0)->m_key;

			for(int iter = 1; iter < size(); ++iter)
			{
				double curMz = at(iter)->m_key;

				tempStep = curMz - prevMz;

				// Spectra are ascending-ordered for the mz value. If not, that's fatal.

				if(tempStep < 0)
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);

				binList->append(tempStep);

				//QString msg = QString("Append step: %1 from computation %2 - %3 with result: %4")
				//.arg(tempStep, 0, 'f', 10)
				//.arg(curMz, 0, 'f', 10)
				//.arg(prevMz, 0, 'f', 10)
				//.arg(curMz - prevMz, 0, 'f', 10);

				//qDebug() << __FILE__ << __LINE__ << msg;

				if(tempStep > greatestStep)
					greatestStep = tempStep;

				prevMz = curMz;
			}

			//qDebug() << __FILE__ << __LINE__
			//<< "Number of elements in binList:" << binList->size();

#if 0

			qDebug() << __FILE__ << __LINE__
				<< "Writing the list of bins in file /tmp/initial-bins.txt";

			QFile file("/tmp/initial-bins.txt");
			file.open(QIODevice::WriteOnly);

			QTextStream fileStream(&file);

			for(int iter = 0; iter < binList->size(); ++iter)
			{

				fileStream << QString("%1 %2\n")
					.arg(binList->at(iter), 0, 'f', 10)
					.arg(0);
			}

			fileStream.flush();
			file.close();

#endif

			// Only set to the member datum if that was not computed already. To set
			// it forcefully, use the other function.

			if(qIsNaN(m_greatestStep))
				m_greatestStep = greatestStep;

			return greatestStep;
		}


	//! Compute the average of the bin size for the bins in \p binList.
	/*!

		\p binList contains all the bin sizes of the sprectrum. A bin is the
		distance between two consecutive key (that is, m/z) values in a spectrum. If
		a spectrum contains n DataPoint instances, then the binList will contain
		(n-1) bins.

		Knowing the average size of the bins is important to establish the
		binnability of the spectrum. The bins need to be roughly of the same size
		throughout all the length of the key (m/z) axis values. 

		That value should then be compared to MassSpectrum::m_maxAvgMzBinSize.

		\param binList list of bins for which the computation is to be performed.

		\return the average of all the values in \p binList.

*/
	double 
		MassSpectrum::binSizeAvg(const QList<double> &binList)
		{
			if(binList.isEmpty())
				return qSNaN();

			int binCount = binList.size();

			double binSizeSum = 0;
			double binSizeAvg = 0;

			for(int iter = 0; iter < binCount; ++iter)
			{
				binSizeSum += binList.at(iter);
			}

			binSizeAvg = binSizeSum / binCount;

			return binSizeAvg;
		}


	//! Compute the standard deviation of the bin size for the bins in \p binList.
	/*!

		\p binList contains all the bin sizes of the sprectrum. A bin is the
		distance between two consecutive m/z values in a spectrum. If a spectrum
		contains n DataPoint instances, then the binList will contain (n-1) bins.

		Knowing the standard deviation of the size of the bins is important to
		establish the binnability of the spectrum. The bins need to be roughly of
		the same size throughout all the length of the m/z axis values. 

		\param binList list of bins for which the computation is to be performed.

		\return the standard deviation of all the values in \p binList.

*/
	double 
		MassSpectrum::binSizeStdDev(const QList<double> &binList)
		{
			if(binList.isEmpty())
				return qSNaN();

			double binCount = binList.size();

			double sizeAvg = binSizeAvg(binList);

			// If a spectrum has not a proper m/z value ladder it will not be
			// binnable. This is typically the case with the spectra obtained from
			// Water's Synapt MS in mobility mode. Below is a rough test.

			double binSizeVarN = 0;

			for(int iter = 0; iter < binCount; ++iter)
			{
				binSizeVarN += (binList.at(iter) - sizeAvg) * (binList.at(iter) - sizeAvg);
			}

			double binSizeVar = binSizeVarN / binCount;

			return sqrt(binSizeVar);
		}


	//! Check that \p binList contains binnable data.
	/*!

		A mass spectrum is considered binnable if all the key (m/z) intervals are
		much less than unity (that is, << MassSpectrum::m_maxAvgMzBinSize) and if
		they are similar (peaks are uniformly distributed along the m/z axis). The
		uniformity of the m/z values is evaluated by means of very simple
		statistics\: the variance must be less than an arbitrary value (0.001, at
		the moment). Also, the m/z interval average must not be greater than 0.1.

		\param binList list of double values to be analyzed

		\return true if the m/z intervals (bins) have an average less than
		MassSpectrum::m_maxAvgMzBinSize and a variance less than
		(MassSpectrum::m_maxAvgMzBinSize / 5).

*/
	bool
		MassSpectrum::isBinnable(const QList<double> &binList)
		{

			// A spectrum is binnable if when looking at all the m/z intervals in it,
			// they are all relatively similar and much less than unity (<< 1).

			if(binList.isEmpty())
				return false;

			double sizeAvg = binSizeAvg(binList);
			double sizeStdDev = binSizeStdDev(binList);

			//QString msg = QString("binCount = %1 - binSizeAvg = %2 - binSizeStdDev = %3\n")
			//.arg(binList.size())
			//.arg(sizeAvg, 0, 'f', 10)
			//.arg(sizeStdDev, 0, 'f', 10);
			//qDebug() << __FILE__ << __LINE__ << msg;

			return (sizeAvg <= MassSpectrum::m_maxAvgMzBinSize &&
					sizeStdDev <= MassSpectrum::m_maxBinSizeStdDev);
		}


	//! Populate \c this MassSpectrum using the m/z values in \p binList
	/*!

		Interate in \p binList and for each m/z value contained in it, create a new
		DataPoint instance with a 0-intensity value and append it to \c this
		MassSpectrum. At the end of the process, \c this MassSpectrum will have a
		list of DataPoint instances, all with 0-intensity values but with the m/z
		values representing the bins.

		\param binList list of m/z values representing the bins

*/
	void
		MassSpectrum::setupBins(const QList<double> &binList)
		{
			if(size())
				qFatal("Fatal error at %s@%d. Program aborted.",
						__FILE__, __LINE__);

			// Sanity check to ensure that the binning setup process was performed in
			// a proper way. Compare the present size of the list of bins with the
			// size that should have been computed earlier.

			if(binList.size() != m_binCount)
			{
				QString msg = QString("binList size: %1 -- m_binCount: %2\n")
					.arg(binList.size()).arg(m_binCount);

				qFatal("Fatal error at %s@%d with msg: %s. Program aborted.",
						__FILE__, __LINE__, msg.toLatin1().data());
			}

			// Seed the mass spectrum with the very first mz value that was found in
			// all the spectra of the QList<MassSpectrum *> used by setupBinnability().
			// That is going to be the very first append. Then we'll append as many
			// data points as there are bins. This way, this spectrum will have as many
			// data point items as there were in the first mass spectrum of the
			// massSpectraList that was used in setupBinnability().  Thus, the for
			// loop then starts at index 1.

			append(new DataPoint(m_binMinMz, 0));
			double previousMzBin = m_binMinMz;

			for(int iter = 0; iter < m_binCount; ++iter)
			{
				double tempMzBin = previousMzBin + binList.at(iter);

				//QString msg = QString("Append bin mz: %1 from computation %2 + %3 with result: %4")
				//.arg(tempMzBin, 0, 'f', 10)
				//.arg(previousMzBin, 0, 'f', 10)
				//.arg(binList.at(iter), 0, 'f', 10)
				//.arg(previousMzBin + binList.at(iter), 0, 'f', 10);

				//qDebug() << __FILE__ << __LINE__ << msg;

				previousMzBin = tempMzBin;

				append(new DataPoint(previousMzBin, 0));
			}

#if 0

			qDebug() << __FILE__ << __LINE__
				<< "Writing the list of bins setup in the mass spectrum in file /tmp/massSpecBins.txt";

			QFile file("/tmp/massSpecBins.txt");
			file.open(QIODevice::WriteOnly);

			QTextStream fileStream(&file);

			// We use the size of *this spectrum as the right border of the bin range.
			// We cannot use m_binCount because its size is one less then the actual
			// *this.size() because we added one extra smallest mz value before
			// looping in binList above. So *this mass spectrum ends having the same
			// number of data points a the intial spectra in the mass spectrum list
			// that was to be combined in this initially empty mass spectrum.
			for(int iter = 0; iter < size(); ++iter)
			{

				fileStream << QString("%1 %2\n")
					.arg(at(iter)->m_key, 0, 'f', 10)
					.arg(0);
			}

			fileStream.flush();
			file.close();

#endif

			//qDebug() << __FILE__ << __LINE__
			//<< "Prepared bins with " << size() << "elements."
			//<< "starting with mz" << first()->m_key
			//<< "ending with mz" << last()->m_key;
		}


	//! Find the index of the bin containing \p mz
	/*!

		Iterates in \c this MassSpectrum and searches the DataPoint instance having
		the \c mz key.

		\param mz m/z value whose corresponding bin is searched.

		\return the index of the DataPoint instance having the \c mz key.

*/
	int
		MassSpectrum::binIndex(double mz)
		{
			//QString msg = QString("Entering binIndex with mz: %1 while m_lastBinIdx = %2 (%3,%4) and m_mzShift = %5")
			//.arg(mz, 0, 'f', 10)
			//.arg(m_lastBinIdx)
			//.arg(at(m_lastBinIdx)->m_key, 0, 'f', 10)
			//.arg(at(m_lastBinIdx)->m_val, 0, 'f', 10)
			//.arg(m_mzShift, 0, 'f', 10);
			//qDebug() << __FILE__ << __LINE__
			//<< msg;

			if(!size())
				return 0;

			if(mz > at(size() - 1)->m_key)
			{
				m_lastBinIdx = size() -1;
				return m_lastBinIdx;
			}

			// FIXME: it is a problem that we get an mz value that is less than the
			// first bin's mz value. Because, during binnability, we set the very
			// first bin's mz value to the smallest of all the mz values of all the
			// mass spectra in the mass spectrum list (see setupBinnability()). So
			// this situation should never occur.
			if(mz <= at(0)->m_key)
			{
				m_lastBinIdx = 0;

				//qFatal("Fatal error at %s@%d -- %s. "
				//"The binning framework was not setup properly. Exiting."
				//"Program aborted.",
				//__FILE__, __LINE__, __FUNCTION__);	
			}

			if(mz < at(m_lastBinIdx)->m_key)
				m_lastBinIdx = 0;

			int lastTestIndex = 0;
			int returnIndex = -1;

			for(int iter = m_lastBinIdx; iter < size(); ++iter)
			{
				double iterBinMz = at(iter)->m_key ;

				//QString msg = QString("Current bin index: %1 (%2,%3")
				//.arg(iter)
				//.arg(iterBinMz, 0, 'f', 10)
				//.arg(at(iter)->m_val, 0, 'f', 10);
				//qDebug() << __FILE__ << __LINE__
				//<< msg;

				if(mz >= iterBinMz)
				{
					if(mz == iterBinMz)
					{
						returnIndex = iter;
						break;
					}
					else
					{
						// If this is the last bin, then we need to return iter.

						if(iter == size())
						{
							returnIndex = iter;
							break;
						}
						else
							lastTestIndex = iter;
					}
				}
				else if(mz < iterBinMz)
				{
					returnIndex = lastTestIndex;
					break;
				}
				else
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);
			}

			// Due to binning, we might be here if mz was greater than the last mz of
			// the spectrum. In this case we just return the last spectrum index.

			if(returnIndex == -1)
			{
				returnIndex = size() - 1;

				qWarning() << __FILE__ << __LINE__
					<< "needed to set returnIndex to size() - 1;"
					<< "mz: " << QString("%1").arg(mz, 0, 'f', 10)
					<< "bin index:" << returnIndex
					<< "with bin's mz:" << QString("%1").arg(at(returnIndex)->m_key, 0, 'f', 10)
					<< "bin count: size():" << size()
					<< "last bin's mz:" << QString("%1").arg(at(size() - 1)->m_key, 0, 'f', 10);
			}

			m_lastBinIdx = returnIndex;

			//qWarning() << __FILE__ << __LINE__
			//<< "mz: " << QString("%1").arg(mz, 0, 'f', 10)
			//<< "bin index:" << returnIndex
			//<< "with bin's mz:" << QString("%1").arg(at(returnIndex)->m_key, 0, 'f', 10)
			//<< "bin count: size():" << size()
			//<< "last bin's mz:" << QString("%1").arg(at(size() - 1)->m_key, 0, 'f', 10);

			//qWarning() << __FILE__ << __LINE__
			//<< "returning" << returnIndex;

			return returnIndex;
		}


	//! Computes the mz shift between \p massSpectrum and \c this MassSpectrum
	/*!

		The shift is computed by making the difference between the keys of the
		first DataPoint instance in \p massSpectrum and \this MassSpectrum instance.

		As a side effect, \c m_mzShift is set to this difference value.

		\param massSpectrum the MassSpectrum to use for the m/z shift calculation
		against \c this MassSpectrum.

		\return a double containing the m/z shift.

*/
	double
		MassSpectrum::determineMzShift(const MassSpectrum &other)
		{
			if(!size() || !other.size())
				return qSNaN();

			m_mzShift = other.first()->m_key - first()->m_key;

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__
			//<< "m_mzShift:" << m_mzShift;

			return m_mzShift;
		}


	//! Computes the mz shift between \p otherMzValue and \c this MassSpectrum
	/*!

		The shift is computed by making the difference between the keys \p
		otherMzValue and \this MassSpectrum instance' first DataPoint instance.

		As a side effect, \c m_mzShift is set to this difference value.

		\param otherMzValue mz value to use for the m/z shift calculation against \c
		this MassSpectrum's first DataPoint instance.

		\return a double containing the m/z shift.

*/	double
	MassSpectrum::determineMzShift(double otherMzValue)
	{

		m_mzShift = otherMzValue - first()->m_key;

		//qDebug() << __FILE__ << __LINE__ << __FUNCTION__
		//<< "m_mzShift:" << m_mzShift;

		return m_mzShift;
	}


	//! Combine \p dataPoint into \c this MassSpectrum using bins.
	/*!

		The combine operation is performed at the bin that corresponds to the key of
		\p dataPoint. The intensity value of the found bin is incremented by the
		intensity of \p dataPoint. If no bin is found, the program crashes.

		Note that if \c this spectrum is empty, the program crashes, because, by
		definition a binned combine operation requires that the bins be already
		available in \c this MassSpectrum.

		The \p dataPoint key is first copied and the copy is modified according
		to this:

		- m_mzShift is systematically added to the \m dataPoint key. By default,
		m_mzShift is 0, so that does not hurt. However, if a shift was computed,
		then it is systematically accounted for.

		- if m_decimalPlaces is not -1, then the decimal places are taken into
		account for the computation.

		Once the steps above have been performed, the bin corresponding to the key
		of \p dataPoint is determined and its value (that is, intensity) is
		incremented by the value of \p dataPoint.

		\param dataPoint DataPoint instance to be combined to \c this MassSpectrum

		\sa combine(const DataPoint &dataPoint)

		\return the index of the updated bin

*/
	int
		MassSpectrum::combineBinned(const DataPoint &dataPoint)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()";

			// Since this combine process is binned, it is absolutely necessary that
			// this spectrum contains data, that is, at least the bins !

			if(!size())
				qFatal("Fatal error at %s@%d -- %s. "
						"In a binned combine operation, this spectrum cannot be empty\n. "
						"Program aborted.", __FILE__, __LINE__, __FUNCTION__);

			// Account for m_mzShift, that was computed (might be 0 or not) in
			// combine(MassSpectrum).

			double mz = dataPoint.m_key + m_mzShift;

			if(m_decimalPlaces != -1)
			{
				mz = ceil((mz * pow(10, m_decimalPlaces)) - 0.49) / pow(10, m_decimalPlaces);

				// if no decimal at all is required: int mz = (double) (int) mz;

				//qDebug("%s %d Rounding from %.5f to %.5f\n",
				//__FILE__, __LINE__,
				//dataPoint.m_key, mz);
			}

			//qDebug() << __FILE__ << __LINE__
			//<< "Calling binIndex with mz:" << QString("%1").arg(dataPoint.m_key, 0, 'f', 10)
			//<< "and mzShift:" << QString("%1").arg(m_mzShift, 0, 'f', 10);

			int binIdx = binIndex(mz);

			if(binIdx == -1)
				qFatal("Fatal error at %s@%d -- %s. "
						"Failed to find the bin for the m/z value of DataPoint. Program aborted.",
						__FILE__, __LINE__, __FUNCTION__);

			//QString msg = QString("must combine (%1,%2) with bin[%3] (%4, %5)")
			//.arg(mz, 0, 'f', 10)
			//.arg(dataPoint.m_val)
			//.arg(binIdx)
			//.arg(at(binIdx)->m_key, 0, 'f', 10)
			//.arg(at(binIdx)->m_val, 0, 'f', 10);
			//qDebug() << __FILE__ << __LINE__ << msg;

			DataPoint *targetDataPoint = at(binIdx);

			targetDataPoint->m_val += dataPoint.m_val;

			//msg = QString("after combination: (mz,i): (%1,%2)")
			//.arg(at(binIdx)->m_key, 0, 'f', 10)
			//.arg(at(binIdx)->m_val, 0, 'f', 10);
			//qDebug() << __FILE__ << __LINE__ << msg;

			return binIdx;
		}


	//! Subtract \p dataPoint from \c this MassSpectrum using bins.
	/*!

		This is the reverse operation of the combineBinned(const DataPoint &dataPoint)
		operation. Here, the value of \p dataPoint has its sign inverted by
		multiplying it by -1. Then combineBinned(DataPoint) is called as for a normal
		combine operation.

		\param dataPoint DataPoint instance to be subtracted from \c this MassSpectrum

		\sa combineBinned(const DataPoint &dataPoint)

		\return an integer containing the index of the updated bin

*/
	int
		MassSpectrum::subtractBinned(const DataPoint &dataPoint)
		{
			DataPoint localDataPoint(dataPoint);

			// Invert the sign of the intensity so that we can reuse the same
			// combineBinned(DataPoint) function as for the positive combine.

			localDataPoint.m_val *= -1;

			return combineBinned(localDataPoint);
		}


	//! Combine MassSpectrum instance \p other into \c this MassSpectrum.
	/*!

		The mass spectrum is combined into \c this MassSpectrum by iterating in all
		the DataPoint instances it contains and calling either combine(const
		DataPoint &) or combineBinned(const DataPoint &) depending on the \c
		m_isBinnable member.

		Note that if m_isBinnable is true, then determineMzShift() is called.

		\param other MassSpectrum instance to be combined to \c this MassSpectrum

		\return an integer containing the number of combined DataPoint instances

*/
	int
		MassSpectrum::combine(const MassSpectrum &other)
		{

			// We receive a spectrum and we need to combine it in this mass spectrum.

			//qDebug() << __FILE__ << __LINE__
			//<< "Starting combination of the spectrum with "
			//<< other.size() << "peaks in it.";

			// Only apply the mz shift if this spectrum has undergone successfully the
			// binning setup.

			if(m_isBinnable)
				determineMzShift(other);

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int iter = 0;

			for(iter = 0; iter < other.size(); ++iter)
			{
				if(m_isBinnable)
					combineBinned(*other.at(iter));
				else
					Trace::combine(*other.at(iter));
			}

			//qDebug() << __FILE__ << __LINE__
			//<< "Done combining a new spectrum in this spectrum, that has now "
			//<< size() << "data points";

			// Return the number of combined data points.
			return iter;
		}


	//! Subtract MassSpectrum instance \p other from \c this MassSpectrum.
	/*!

		This is the reverse operation of combine(const MassSpectrum &other).

*/
	int
		MassSpectrum::subtract(const MassSpectrum &other)
		{

			// We receive a spectrum and we need to subtract it from this mass spectrum.

			// Only apply the mz shift if this spectrum has undergone successfully the
			// binning setup.

			if(m_isBinnable)
				determineMzShift(other);

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int iter = 0;

			for(iter = 0; iter < other.size(); ++iter)
			{
				if(m_isBinnable)
					subtractBinned(*other.at(iter));
				else
					Trace::subtract(*other.at(iter));
			}

			// Return the number of subtracted data points.
			return iter;
		}


	//! Combine MassSpectrum instance \p other into \c this MassSpectrum.
	/*!

		Same as combine(const MassSpectrum &other) unless that only the
		DataPoint instances having a m/z value comtained in the [\p mzStart -- \p
		mzEnd] range are taken into account.

		\param other MassSpectrum instance to be combined to \c this MassSpectrum

		\param mzStart left value of the m/z range

		\param mzEnd right value of the m/z range

		\return an integer containing the number of combined DataPoint instances

*/
	int
		MassSpectrum::combine(const MassSpectrum &other,
				double mzStart, double mzEnd)
		{
			// We receive a spectrum and we need to combine this spectrum in this mass
			// spectrum. However, we need to check each individual DataPoint of the
			// MassSpectrum instance for their m/z value, because it can only be
			// combined if its value is in between both mzStart and mzEnd arguments to
			// this function.

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()"
			//<< "Starting combination of the spectrum with "
			//<< other.size() << "peaks in it.";

			// Only apply the mz shift if this spectrum has undergone successfully the
			// binning setup.

			if(m_isBinnable)
				determineMzShift(other);

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int count  = 0;

			for(int iter = 0; iter < other.size(); ++iter)
			{
				DataPoint *iterDataPoint = other.at(iter);

				if(iterDataPoint->m_key >= mzStart && iterDataPoint->m_key <= mzEnd)
				{
					//QString msg = QString("Range: [%1-%2] - this data point is in the range: mz %3\n")
					//.arg(mzStart, 0, 'f', 10)
					//.arg(mzEnd, 0, 'f', 10)
					//.arg(iterDataPoint->m_key, 0, 'f', 10);
					//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
					//<< msg; 

					if(m_isBinnable)
						combineBinned(*iterDataPoint);
					else
						Trace::combine(*iterDataPoint);

					++count;
				}
				// Because the spectra are ordered, if the currently iterated data point
				// has a mass greater than the accepted mzEnd, then break the loop as
				// this cannot be any better later.
				else if(iterDataPoint->m_key > mzEnd)
					break;
			}

			// qDebug() << __FILE__ << __LINE__
			// << "Done combining a new spectrum in this spectrum, current with "
			// << size() << "data points";

			return count;
		}


	//! Subtract MassSpectrum instance \p other from \c this MassSpectrum.
	/*!

		This is the reverse operation of combine(const MassSpectrum &other,
		double mzStart, double mzEnd).

*/
	int
		MassSpectrum::subtract(const MassSpectrum &other,
				double mzStart, double mzEnd)
		{
			// qDebug() << __FILE__ << __LINE__ << __FUNCTION__

			// Only apply the mz shift if this spectrum has undergone successfully the
			// binning setup.

			if(m_isBinnable)
				determineMzShift(other);

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int count  = 0;

			for(int iter = 0; iter < other.size(); ++iter)
			{
				DataPoint *iterDataPoint = other.at(iter);

				if(iterDataPoint->m_key >= mzStart && iterDataPoint->m_key <= mzEnd)
				{

					if(m_isBinnable)
						subtractBinned(*iterDataPoint);
					else
						Trace::subtract(*iterDataPoint);

					++count;
				}
				// Because the spectra are ordered, if the currently iterated data point
				// has a mass greater than the accepted mzEnd, then break the loop as
				// this cannot be any better later.
				else if(iterDataPoint->m_key > mzEnd)
					break;
			}
			// qDebug() << __FILE__ << __LINE__
			// << "Done combining a new spectrum in this spectrum, current with "
			// << size() << "data points";

			// Return the number of combined data points.
			return count;
		}


	//! Combine the MassSpectrum instances of \p massSpectra into \c this MassSpectrum
	/*!

		The workings of this function are the following:

		- if the \p massSpectra list is empty, return 0

		- if \c this MassSpectrum is empty, then setup the binning framework (call
		to setupBinnability()).

		- iterate in the \p massSpectra list of MassSpectrum instances and for each
		instance call combine(const MassSpectrum &). Note that the combine() call
		will check if the combine operation should take advantage of binning or not.

		\param massSpectra list of MassSpectrum instances to be combined to \c this
		MassSpectrum

		\return the number of combined MassSpectrum instances.

		\sa combine(const MassSpectrum &)
		*/
	int
		MassSpectrum::combine(const QList<MassSpectrum *> &massSpectra)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()";

			// We receive a list of mass spectra and we need to combine all the
			// spectra thereof in this mass spectrum.

			if(massSpectra.size() == 0)
				return 0;

			//qDebug() << __FILE__ << __LINE__
			//<< "Starting combination of " << massSpectra.size()
			//<< "mass spectra into this mass spectrum";

			// This function might be called recursively with various mass spectrum
			// lists. We only need to seed the bin combination data if that has not
			// been done already, that is this mass spectrum is empty (binning
			// actually fills-in the very first set of (m/z,i) pairs.

			if(!size())
				if(!setupBinnability(massSpectra))
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);

			MassSpectrum *p_spectrum = Q_NULLPTR;
			int specCount = massSpectra.size();

			int count = 0;

			for(int iter = 0; iter < specCount; ++iter)
			{
				p_spectrum = massSpectra.at(iter);

				combine(*p_spectrum);

				++count;

#if 0
				QString fileName = QString("/tmp/unranged-combined-spectrum-after-%1-combinations.txt")
					.arg(iter + 1);

				QFile file(fileName);

				file.open(QIODevice::WriteOnly);

				QTextStream fileStream(&file);

				for(int iter = 0; iter < size(); ++iter)
				{
					fileStream << QString("%1 %2\n")
						.arg(at(iter)->m_key, 0, 'f', 10)
						.arg(at(iter)->m_val, 0, 'f', 10);
				}

				fileStream.flush();
				file.close();
#endif
			}

			// qDebug() << __FILE__ << __LINE__
			// << "Effectively combined " << combinedSpectra
			// << "in this single spectrum for a total of "
			// << size() << "data points";

#if 0
			QString fileName = "/tmp/unranged-combined-spectrum-after-finishing-combinations.txt";

			QFile file(fileName);

			file.open(QIODevice::WriteOnly);

			QTextStream fileStream(&file);

			for(int iter = 0; iter < size(); ++iter)
			{
				fileStream << QString("%1 %2\n")
					.arg(at(iter)->m_key, 0, 'f', 10)
					.arg(at(iter)->m_val, 0, 'f', 10);
			}

			fileStream.flush();
			file.close();
#endif

			return count;
		}


	//! Subtract \p massSpectra from \c this MassSpectrum.
	/*!

		This is the reverse operation of combine(const QList<MassSpectrum *>
		&massSpectra).

*/
	int
		MassSpectrum::subtract(const QList<MassSpectrum *> &massSpectra)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ ;

			if(massSpectra.size() == 0)
				return 0;

			// This function might be called recursively with various mass spectrum
			// lists. We only need to seed the bin combination data if that has not
			// been done already, that is this mass spectrum is empty (binning
			// actually fills-in the very first set of (m/z,i) pairs.

			if(!size())
				if(!setupBinnability(massSpectra))
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);

			MassSpectrum *p_spectrum = Q_NULLPTR;
			int specCount = massSpectra.size();

			int count = 0;

			for(int iter = 0; iter < specCount; ++iter)
			{
				p_spectrum = massSpectra.at(iter);

				subtract(*p_spectrum);

#if 0
				QString fileName = QString("/tmp/unranged-combined-spectrum-after-%1-combinations.txt")
					.arg(iter + 1);

				QFile file(fileName);

				file.open(QIODevice::WriteOnly);

				QTextStream fileStream(&file);

				for(int iter = 0; iter < size(); ++iter)
				{
					fileStream << QString("%1 %2\n")
						.arg(at(iter)->m_key, 0, 'f', 10)
						.arg(at(iter)->m_val, 0, 'f', 10);
				}

				fileStream.flush();
				file.close();
#endif

				++count;
			}
			// qDebug() << __FILE__ << __LINE__
			// << "Effectively combined " << combinedSpectra
			// << "in this single spectrum for a total of "
			// << size() << "data points";

#if 0
			QString fileName = "/tmp/unranged-combined-spectrum-after-finishing-combinations.txt";

			QFile file(fileName);

			file.open(QIODevice::WriteOnly);

			QTextStream fileStream(&file);

			for(int iter = 0; iter < size(); ++iter)
			{
				fileStream << QString("%1 %2\n")
					.arg(at(iter)->m_key, 0, 'f', 10)
					.arg(at(iter)->m_val, 0, 'f', 10);
			}

			fileStream.flush();
			file.close();
#endif

			return count;
		}

	//! Combine the MassSpectrum instances of \c massSpectra into \c this MassSpectrum
	/*!

		This function acts like

		combine(const QList<MassSpectrum *> &massSpectra) with the difference that it calls

		combine(const MassSpectrum &massSpectrum, double mzStart, double mzEnd) to
		only account for DataPoint instances in the spectra that have their m/z value
		contained in the [\p mzStart -- \p mzEnd] range.

		\param massSpectra list of MassSpectrum instances to be combined to \c this
		MassSpectrum

		\param mzStart start value of the acceptable m/z range

		\param mzEnd end value of the acceptable m/z range

		\return an int containing the number of combined MassSpectrum instances.

		\sa combine(const QList<MassSpectrum *> &massSpectra)

*/
	int
		MassSpectrum::combine(const QList<MassSpectrum *> &massSpectra,
				double mzStart, double mzEnd)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
			//<< "";

			// We receive a list of mass spectra, that is a list of (list of
			// DataPoint*) and wee need to combine all the spectra thereof in this mass
			// spectrum. However, we need to check each individual DataPoint of the
			// MassSpectrum instances for their m/z value, because it can only be
			// combined if its value is in between both mzStart and mzEnd arguments to
			// this function.

			if(massSpectra.size() == 0)
				return 0;

			//qDebug() << __FILE__ << __LINE__
			//<< "Starting combination of " << massSpectra.size()
			//<< "mass spectra into this mass spectrum";

			if(!size())
				if(!setupBinnability(massSpectra))
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);

			int count = 0;

			for(int iter = 0; iter < massSpectra.size(); ++iter)
			{
				MassSpectrum *p_spectrum = massSpectra.at(iter);

				combine(*p_spectrum, mzStart, mzEnd);

				// qDebug() << __FILE__ << __LINE__
				// << "Currently processed mass spectrum has"
				// << iterMassSpectrum->size() << "peaks";

				++count;
			}

			// qDebug() << __FILE__ << __LINE__
			// << "Effectively combined " << combinedSpectra
			// << "in this single spectrum for a total of "
			// << size() << "data points";

#if 0
			QString fileName = "/tmp/mzranged-combined-spectrum-after-finishing-combinations.txt";

			QFile file(fileName);

			file.open(QIODevice::WriteOnly | QIODevice::Truncate);

			QTextStream fileStream(&file);

			for(int iter = 0; iter < size(); ++iter)
			{
				fileStream << QString("%1 %2\n")
					.arg(at(iter)->m_key, 0, 'f', 10)
					.arg(at(iter)->m_val, 0, 'f', 10);
			}

			fileStream.flush();
			file.close();
#endif

			return count;
		}


	//! Subtract \p massSpectra from \c this MassSpectrum.
	/*!

		This is the reverse operation of combine(const QList<MassSpectrum *> &massSpectra,
		double mzStart, double mzEnd).

*/
	int
		MassSpectrum::subtract(const QList<MassSpectrum *> &massSpectra,
				double mzStart, double mzEnd)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ ;

			if(massSpectra.size() == 0)
				return 0;

			if(!size())
				if(!setupBinnability(massSpectra))
					qFatal("Fatal error at %s@%d. Program aborted.",
							__FILE__, __LINE__);

			int count = 0;

			for(int iter = 0; iter < massSpectra.size(); ++iter)
			{
				MassSpectrum *p_spectrum= massSpectra.at(iter);

				subtract(*p_spectrum, mzStart, mzEnd);

				// qDebug() << __FILE__ << __LINE__
				// << "Currently processed mass spectrum has"
				// << iterMassSpectrum->size() << "peaks";

				++count;
			}

			// qDebug() << __FILE__ << __LINE__
			// << "Effectively combined " << combinedSpectra
			// << "in this single spectrum for a total of "
			// << size() << "data points";

#if 0
			QString fileName = "/tmp/mzranged-combined-spectrum-after-finishing-combinations.txt";

			QFile file(fileName);

			file.open(QIODevice::WriteOnly | QIODevice::Truncate);

			QTextStream fileStream(&file);

			for(int iter = 0; iter < size(); ++iter)
			{
				fileStream << QString("%1 %2\n")
					.arg(at(iter)->m_key, 0, 'f', 10)
					.arg(at(iter)->m_val, 0, 'f', 10);
			}

			fileStream.flush();
			file.close();
#endif

			return count;
		}


	//! Combine the mass spectrum represented by \p mzList and \iList in \c this MassSpectrum.
	/*!

		The \p mzList and \p iList represent a mass spectrum. Both the lists must
		contain at least one value and have the same number of items. If \p mzList
		is empty, the function returns immediately. If both lists do not have the
		same size, the program crashes.

		Note that if \c this MassSpectrum is binnable, then the m/z shift is
		determined by calling determineMzShift().

		\param mzList list of double values representing the m/z values of the mass
		spectrum.

		\param iList list of double values representing the intensity values of the
		mass spectrum.

		\return -1 if \p mzList is empty, otherwise the number of combined (m/z,i) pairs.

*/
	int
		MassSpectrum::combine(const QList<double> &mzList, const QList<double> &iList)
		{

			// We get two lists of mz and i, typically out of the data file, and we
			// need to construct (or continue constructing) a mass spectrum. For each
			// mz in the mzList, we iterate in the spectrum and we check if that mz
			// exists. If so the intensity of the found DataPoint is increased. If not
			// a new DataPoint is inserted at the proper place with the corresonding
			// intensity from iList.

			// Sanity check

			int listSize = mzList.size();
			if(listSize < 1)
				return -1;

			if(listSize != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			if(m_isBinnable)
				determineMzShift(mzList.first());

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int count = 0;

			for(int iter = 0; iter < listSize; ++iter)
			{
				if(m_isBinnable)
					count += combineBinned(DataPoint(mzList.at(iter), iList.at(iter)));
				else
					count += Trace::combine(DataPoint(mzList.at(iter), iList.at(iter)));
			}

			return count;
		}


	//! Subtract mass data materialized in \p mzList and \p iList from \c this MassSpectrum.
	/*!

		This is the reverse operation of combine(const QList<double> &mzList, const
		QList<double> &iList).

*/
	int
		MassSpectrum::subtract(const QList<double> &mzList, const QList<double> &iList)
		{

			// Sanity check

			int listSize = mzList.size();
			if(listSize < 1)
				return -1;

			if(listSize != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			if(m_isBinnable)
				determineMzShift(mzList.first());

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int count = 0;

			for(int iter = 0; iter < listSize; ++iter)
			{
				if(m_isBinnable)
					count += subtractBinned(DataPoint(mzList.at(iter), iList.at(iter)));
				else
					count += Trace::subtract(DataPoint(mzList.at(iter), iList.at(iter)));
			}

			return count;
		}


	//! Combine the mass spectrum represented by \p mzList and \iList in \c this MassSpectrum.
	/*!

		Works like combine(const QList<double> &mzList, const QList<double> &iList)
		unless the (m/z,i) pairs out of the lists are accounted for only if the m/z
		value is contained in the [\p mzStart -- \p mzEnd] range.

		\param mzList list of double values representing the m/z values of the mass
		spectrum

		\param iList list of double values representing the intensity values of the
		mass spectrum

		\param mzStart start value of the acceptable m/z range

		\param mzEnd end value of the acceptable m/z range

		\return -1 if \p mzList is empty, otherwise the number of combined (m/z,i) pairs.

		\sa combine(const QList<double> &mzList, const QList<double> &iList).

*/
	int
		MassSpectrum::combine(const QList<double> &mzList, const QList<double> &iList,
				double mzStart, double mzEnd)
		{

			// Sanity check

			int listSize = mzList.size();
			if(listSize < 1)
				return -1;

			if(listSize != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			if(m_isBinnable)
				determineMzShift(mzList.first());

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int count = 0;

			for(int iter = 0; iter < listSize; ++iter)
			{
				double mz = mzList.at(iter);

				if(mz >= mzStart && mz <= mzEnd)
				{
					if(m_isBinnable)
						count += combineBinned(DataPoint(mzList.at(iter), iList.at(iter)));
					else
						count += Trace::combine(DataPoint(mzList.at(iter), iList.at(iter)));
				}
			}

			return count;
		}


	//! Subtract mass data materialized in \p mzList and \p iList from \c this MassSpectrum.
	/*!

		This is the reverse operation of combine(const QList<double> &mzList, const
		QList<double> &iList, double mzStart, double mzEnd).

*/
	int
		MassSpectrum::subtract(const QList<double> &mzList, const QList<double> &iList,
				double mzStart, double mzEnd)
		{

			// Sanity check

			int listSize = mzList.size();
			if(listSize < 1)
				return -1;

			if(listSize != iList.size())
				qFatal("Fatal error at %s@%d. Program aborted.", __FILE__, __LINE__);

			if(m_isBinnable)
				determineMzShift(mzList.first());

			// Reset the index anchors
			m_lastBinIdx = 0;
			m_lastModifIdx = 0;

			int count = 0;

			for(int iter = 0; iter < listSize; ++iter)
			{
				double mz = mzList.at(iter);

				if(mz >= mzStart && mz <= mzEnd)
				{
					if(m_isBinnable)
						count += subtractBinned(DataPoint(mzList.at(iter), iList.at(iter)));
					else
						count += Trace::subtract(DataPoint(mzList.at(iter), iList.at(iter)));
				}
			}

			return count;
		}


	//! Calculate the total ion current of \c this MassSpectrum.
	/*!

		This function calls Trace::valSum().

*/
	double 
		MassSpectrum::tic()
		{
			return Trace::valSum();
		}


	//! Calculate the total ion current of \c this MassSpectrum.
	/*!

		This function calls Trace::valSum().

*/
	double 
		MassSpectrum::tic(double mzStart, double mzEnd)
		{
			return Trace::valSum(mzStart, mzEnd);
		}


	//! Compute the total ion current of a MassSpectrumList instance
	/*!

		The \p massSpectra list of MassSpectrum instances is iterated in and
		for each MassSpectrum instance the TIC is computed and added to a local
		variable that is returned.

		\param massSpectra list of MassSpectrum instances for which the total
		TIC value is to be computed.

		\return a double value containing the sum of all the total ion current
		values of the whole set of MassSpectrum instances in \p massSpectra.

		\sa tic()
		\sa tic(double mzStart, double mzEnd)

*/
	double
		MassSpectrum::tic(const QList<MassSpectrum *> &massSpectra)
		{
			double totalTic = 0;

#pragma omp parallel for reduction (+:totalTic)
			for(int jter = 0; jter < massSpectra.size(); ++jter)
			{
				totalTic += massSpectra.at(jter)->tic();
			}

			return totalTic;
		}



	//! Fill in the characteristics of \p massSpectra
	/*!

		The set of MassSpectrum instances stored in \p massSpectra is analyzed to
		extract characteristic data and to set them into the variables pointed to by
		the parameters:

		\param minMz the minimum key (m/z) of all the spectra. If nullptr, that bit
		of information is not set

		\param maxMz the maximum key (m/z) of all the spectra. If nullptr, that bit
		of information is not set

*/
	void 
		MassSpectrum::mzMinMax(const QList<MassSpectrum *> &massSpectra,
				double *minMz, double *maxMz)
		{

			MassSpectrum *massSpectrum = Q_NULLPTR;

			double finalMinMz = std::numeric_limits<double>::max();
			double finalMaxMz = 0;

			for(int iter = 0; iter < massSpectra.size(); ++iter)
			{

				massSpectrum = massSpectra.at(iter);

				if(maxMz != Q_NULLPTR)
				{
					double tempMaxMz = massSpectrum->last()->key();
					if(tempMaxMz > finalMaxMz)
						finalMaxMz = tempMaxMz;
				}

				if(minMz != Q_NULLPTR)
				{
					double tempMinMz = massSpectrum->first()->key();
					if(tempMinMz < finalMinMz)
						finalMinMz = tempMinMz;
				}
			}
			// End of 
			// for(int iter = 0; iter < size(); ++iter)

			if(minMz != Q_NULLPTR)
				*minMz = finalMinMz;
			if(maxMz != Q_NULLPTR)
				*maxMz = finalMaxMz;
		}


	//! Check if all the MassSpectrum instances have the same length.
	/*!

		The computation here is statistic: the set of MassSpectrum instances is
		iterated in and for each spectrum, its size (number of MassPeak instances)
		is recorded. Then, the standard deviation is computed.  \c .

		\return false if the standard deviation is greater than (avg / 100), true
		otherwise.

*/
	bool 
		MassSpectrum::areSpectraSameLength(const QList<MassSpectrum *> &massSpectra)
		{
			int specCount = massSpectra.size();

			double specSizeSum = 0;
			double specSizeAvg = 0;

			for(int iter = 0; iter < specCount; ++iter)
			{
				specSizeSum += massSpectra.at(iter)->size();
			}

			specSizeAvg = specSizeSum / specCount;

			double specSizeVarN = 0;

			for(int iter = 0; iter < specCount; ++iter)
			{
				specSizeVarN += (massSpectra.at(iter)->size() - specSizeAvg) * 
					(massSpectra.at(iter)->size() - specSizeAvg);
			}

			double specSizeVar = specSizeVarN / specCount;

			double specSizeStdDev = sqrt(specSizeVar);

			//QString msg = QString("specCount = %1 - specSizeSum: %2 - specSizeAvg = %3 - specSizeVarN = %4"
			//" - specSizeVar = %5 - specSizeStdDev = %6\n")
			//.arg(specCount)
			//.arg(specSizeSum, 0, 'f', 7)
			//.arg(specSizeAvg, 0, 'f', 7)
			//.arg(specSizeVarN, 0, 'f', 7)
			//.arg(specSizeVar, 0, 'f', 7)
			//.arg(specSizeStdDev, 0, 'f', 7);
			//qWarning() << __FILE__ << __LINE__
			//<< msg;

			if(specSizeStdDev > specSizeAvg / 100)
				return false;

			return true;
		}



} // namespace msXpSlibmass
