/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */


#ifndef HISTORY_HPP
#define HISTORY_HPP

/////////////////////// Qt includes
#include <QList>
#include <QDateTime>
#include <QMap>


/////////////////////// Local includes

namespace msXpSmineXpert
{


	//! Enum to define the kind of mass data integration
	/*!

		Analyzing mass spectrometric data involves performing integration of many
		different kinds. For example, once can integrate all the mass spectra
		acquired between two retention time values and produce a mass spectrum; or
		one can integrate between the same retention time values but produce not a
		mass spectrum, but a drift spectrum. Another integration could be to make a
		drift spectrum with all the spectra that were acquired between two m/z
		values in a given mass spectrum, which would mean "show me the drift times
		of all the ions that were observed in this spectrum between m/z value and
		m/z value. The members of this enumeration serve to define more or less
		precisely the kind of integration that is being performed. This information
		is important to document what a given trace (be that trace a TIC
		chromatogram, a XIC chromatogram, a mass spectrum or a drift spectrum)
		contains with respect to the initial data contained in the mass spectrometry
		data file because the user will want to refer to that information to
		actually mine the data.

*/
	enum IntegrationType
	{
		NOT_SET = 0x0000,

		DATA_TO_RT = 1 << 0,
		/*!< Integrate from mass data file to retention time (TIC chromatogram) */

		DATA_TO_MZ = 1 << 1,
		/*!< Integrate from mass data file to retention time (TIC chromatogram) */

		DATA_TO_DT = 1 << 2,
		/*!< Integrate from mass data file to retention time (TIC chromatogram) */

		RT_TO_MZ = 1 << 3,
		/*!< Integrate from TIC or XIC chromatogram to mass spectrum */

		RT_TO_DT = 1 << 4,
		/*!< Integrate from TIC or XIC chromatogram to drift spectrum */

		RT_TO_TIC_INT = 1 << 5,
		/*!< Integrate from TIC chrom range to TIC intensity single value */

		MZ_TO_RT = 1 << 6,
		/*!< Integrate from mass spectrum to XIC chromatogram (retention time) */

		MZ_TO_MZ = 1 << 7,
		/*!< Integrate from mass spectrum to mass spectrum*/

		MZ_TO_DT = 1 << 8,
		/*!< Integrate from mass spectrum to drift spectrum*/

		MZ_TO_TIC_INT = 1 << 9,
		/*!< Integrate from mass spectrum range to TIC intensity single value */

		DT_TO_RT = 1 << 10,
		/*!< Integrate from drift spectrum to XIC (retention time) */

		DT_TO_MZ = 1 << 11,
		/*!< Integrate from drift spectrum to mass spectrum */

		DT_TO_DT = 1 << 12,
		/*!< Integrate from drift spectrum to drift spectrum */

		DT_TO_TIC_INT = 1 << 13,
		/*!< Integrate from drift spectrum range to TIC intensity single value */

		RT_TO_ANY = (RT_TO_MZ | RT_TO_DT | RT_TO_TIC_INT),

		MZ_TO_ANY = (MZ_TO_RT | MZ_TO_MZ | MZ_TO_DT | MZ_TO_TIC_INT),

		DT_TO_ANY = (DT_TO_RT | DT_TO_MZ | DT_TO_DT | DT_TO_TIC_INT),

		ANY_TO_RT = (DATA_TO_RT | MZ_TO_RT | DT_TO_RT),

		ANY_TO_MZ = (DATA_TO_MZ | RT_TO_MZ | MZ_TO_MZ | DT_TO_MZ),

		ANY_TO_DT = (DATA_TO_DT | RT_TO_DT | MZ_TO_DT | DT_TO_DT),

		ANY_TO_TIC_INT = (RT_TO_TIC_INT | MZ_TO_TIC_INT | DT_TO_TIC_INT),
	};


	//! The IntegrationRange structure describes a range of integration
	/*!

		When performing a data integration, like creating a mass spectrum starting
		from a TIC chromatogram, the user typically selects a TIC chromatogram
		retention time range for which to combine the mass spectra acquired during
		that time range into a single spectrum.

		Note that the \c start and \c end values of the range are to be considered
		part of the range, that is the range is [\c start -- \c end].

*/
	struct IntegrationRange
	{
		double start; 
		/*!< start (left) value of the range */

		double end;
		/*!< end (right) value of the range */
	};


	//! The IntegrationTypeDesc structure defines a description of a data integration
	/*!

		The IntegrationTypeDesc structure helps maintaining a detailed record of the
		various integrations that are performed in the course of data mining of mass
		spectrometry data. This helper structure gathers two textual representations
		of a given data integration step.

*/
	struct IntegrationTypeDesc
	{
		QString detailed;
		/*!< Detailed textual representation of an integration step */

		QString brief;
		/*!< Brief textual representation of an integration step */
	};


	//! The HistoryItem class provides a single element that can be part of a History
	/*!

		When the user performs data integration in the course of mass spectrometric
		data analysis, it is necessary that the various integration steps be stored
		as meaningful data bits, both for the program to know where a given trace
		(mass spec, drift spec or chromatogram, for exemple) originates from and for
		the user to understand the various data mining results that are a result of
		the way the data integrations were performed in sequence during the data
		mining session.

		This class provides two map member data to put in correspondence:

		-	IntegrationType keys with IntegrationTypeDesc pointers

		- IntegrationType keys with IntegrationRange pointers 

*/
	class HistoryItem
	{

		friend class History;

		private:

		//! Date and time stamp of this HistoryItem instance 
		QDateTime m_dateTime;

		//! Map relating IntegrationType keys to pointers to IntegrationTypeDesc values
		/*! 

			This map is initialized upon construction of the HistoryItem instance
			and is regarded as, somehow, a "static" member map because it never
			changes from an instance and the others.

			Note that the IntegrationTypeDesc instances are owned by \c this
			instance and are destroyed by the destructor.

*/
		QMap<int /* IntegrationType */, IntegrationTypeDesc *> m_intTypeDescMap;

		//! Map relating IntegrationType keys to pointers to IntegrationRange values
		/*!

			Note that the IntegrationRange instances are owned by \c this instance
			and are destroyed by the destructor.

*/
		QMap<int /* IntegrationType */, IntegrationRange *> m_integrationMap;

		public:

		HistoryItem();
		HistoryItem(const HistoryItem &other);
		HistoryItem &operator=(const HistoryItem &other);
		~HistoryItem();

		void initializeIntegrationTypes();

		bool operator==(const HistoryItem &other) const;
		bool contains(int intType, const IntegrationRange &intRange) const;

		void setDateTime();
		QDateTime dateTime() const;

		void newIntegrationRange(int intType, double start, double end);

		bool hasIntegrationType(int intType);
		IntegrationRange *integrationRange(int intType) const;
		int integrationType(IntegrationRange *integrationRange);

		QList<int> integrationTypes() const;
		QList<IntegrationRange *> integrationRanges() const;

		bool isValid() const;

		QString asText(bool brief = false) const;
	};


	//! The History class provides a means to store an integration history
	/*!

		When the user performs data integration in the course of mass spectrometric
		data analysis, it is necessary that the various integration steps be stored
		as meaningful data bits, both for the program to know where a given trace
		(mass spec, drift spec or chromatogram, for exemple) originates from and for
		the user to understand the various data mining results that are a result of
		the way the data integrations were performed in sequence during the data
		mining session.

		The History class provides a means to store a list of HistoryItem instances
		(heap-allocated instances). The use of the History class is to typically
		attach one instance to each plot widget used in the mineXpert program so
		that any given widget knows the history of the data it plots.

		\sa HistoryItem

*/
	class History
	{

		private:

			//! List of HistoryItem instances
			/*!

				The instances listed in the list are heap-allocated and the stored data
				are in fact pointers to HistoryItem instances.

*/
			QList<HistoryItem *> m_historyItemList;

		public:

			History();
			History(const History &other);
			History &operator=(const History &other);
			~History();

			void freeList();

			void copyHistory(const History &history, bool noDuplicates = true);
			int appendHistoryItem(HistoryItem *item);
			int copyHistoryItem(const HistoryItem &item);

			HistoryItem *firstHistoryItem(int integrationType,
					IntegrationRange **integrationRange = Q_NULLPTR);

			HistoryItem *lastHistoryItem(int integrationType,
					IntegrationRange **integrationRange = Q_NULLPTR);

			const HistoryItem *newestHistoryItem() const;

			int containsHistoryItem(const HistoryItem &item);

			bool innermostRtRange(double *start, double *end) const;
			bool innermostMzRange(double *start, double *end) const;
			bool innermostDtRange(double *start, double *end) const;
			bool innermostRange(int integrationType, double *start,
					double *end) const;

			bool innermostRange(double *start, double *end) const;

			QString innermostRangesAsText() const;

			bool isValid() const;

			QString asText(const QString &header = QString(),
					const QString &footer = QString(), bool brief = false) const;
	};


} // namespace msXpSmineXpert

#endif /* HISTORY_HPP */
