///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/utilities/PathManager.h>
#include <core/plugins/PluginManager.h>
#include <core/scene/objects/SceneObject.h>
#include <core/scene/objects/ModifiedObject.h>
#include <core/scene/objects/Modifier.h>
#include <core/scene/ObjectNode.h>
#include <core/undo/UndoManager.h>
#include <core/actions/ActionManager.h>
#include "ModifyCommandPage.h"
#include "ModifierStack.h"

namespace Core {

/******************************************************************************
* Initializes the modify page.
******************************************************************************/
ModifyCommandPage::ModifyCommandPage() : CommandPanelPage()
{
	scanInstalledModifierClasses();

	stack = new ModifierStack(this);

	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setContentsMargins(2,2,2,2);
	layout->setSpacing(0);

	QHBoxLayout* subLayout = new QHBoxLayout();
	subLayout->setContentsMargins(0,0,0,0);
	subLayout->setSpacing(0);

	nodeNameBox = new QLineEdit(this);
	nodeNameBox->setEnabled(false);
	// Show this only in experimental mode.
	nodeNameBox->setVisible(APPLICATION_MANAGER.experimentalMode());
	subLayout->addWidget(nodeNameBox, 1);
	connect(nodeNameBox, SIGNAL(editingFinished()), this, SLOT(onNodeNameEntered()));

	nodeDisplayColorPicker = new ColorPickerWidget();
	nodeDisplayColorPicker->setEnabled(false);
	nodeDisplayColorPicker->setColor(Color(palette().color(QPalette::Window)));
	// Show this only in experimental mode.
	nodeDisplayColorPicker->setVisible(APPLICATION_MANAGER.experimentalMode());
	subLayout->addWidget(nodeDisplayColorPicker);
	connect(nodeDisplayColorPicker, SIGNAL(colorChanged()), this, SLOT(onNodeDisplayColorPicked()));

	layout->addLayout(subLayout);

	modifierSelector = new QComboBox();
	layout->addSpacing(4);
    layout->addWidget(modifierSelector);
    connect(modifierSelector, SIGNAL(activated(int)), this, SLOT(onModifierAdd(int)));

	class ModifierStackListView : public QListView {
	public:
		ModifierStackListView(QWidget* parent) : QListView(parent) {}
		virtual QSize sizeHint() const { return QSize(256, 130); }
	};

	QSplitter* splitter = new QSplitter(Qt::Vertical);
	splitter->setChildrenCollapsible(false);

	QWidget* upperContainer = new QWidget();
	splitter->addWidget(upperContainer);
	subLayout = new QHBoxLayout(upperContainer);
	subLayout->setContentsMargins(0,0,0,0);
	subLayout->setSpacing(0);

	stackBox = new ModifierStackListView(upperContainer);
	stackBox->setModel(stack->listModel());
	connect(stackBox->selectionModel(), SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)), this, SLOT(onModifierStackSelectionChanged()));
	connect(stackBox, SIGNAL(doubleClicked(const QModelIndex&)), this, SLOT(onModifierStackDoubleClicked(const QModelIndex&)));
	layout->addSpacing(4);
	subLayout->addWidget(stackBox);

	QToolBar* editToolbar = new QToolBar(this);
	editToolbar->setOrientation(Qt::Vertical);
#ifndef Q_WS_MAC
	editToolbar->setStyleSheet("QToolBar { padding: 0px; margin: 0px; border: 0px none black; }");
#endif
	subLayout->addWidget(editToolbar);

	ActionProxy* deleteModifierAction = ACTION_MANAGER.addAction(new Action(ACTION_MODIFIER_DELETE), tr("Delete Modifier"), ":/core/command_panel/delete_modifier.png");
	connect(deleteModifierAction, SIGNAL(triggered(bool)), this, SLOT(onDeleteModifier()));
	editToolbar->addAction(deleteModifierAction);

	editToolbar->addSeparator();

	ActionProxy* moveModifierUpAction = ACTION_MANAGER.addAction(new Action(ACTION_MODIFIER_MOVE_UP), tr("Move Modifier Up"), ":/core/command_panel/modifier_move_up.png");
	connect(moveModifierUpAction, SIGNAL(triggered(bool)), this, SLOT(onModifierMoveUp()));
	editToolbar->addAction(moveModifierUpAction);
	ActionProxy* moveModifierDownAction = ACTION_MANAGER.addAction(new Action(ACTION_MODIFIER_MOVE_DOWN), tr("Move Modifier Down"), ":/core/command_panel/modifier_move_down.png");
	connect(moveModifierDownAction, SIGNAL(triggered(bool)), this, SLOT(onModifierMoveDown()));
	editToolbar->addAction(moveModifierDownAction);

	editToolbar->addSeparator();

	ActionProxy* toggleModifierStateAction = ACTION_MANAGER.addAction(new Action(ACTION_MODIFIER_TOGGLE_STATE), tr("Enable/Disable Modifier"));
	toggleModifierStateAction->setCheckable(true);
	QIcon toggleStateActionIcon(QString(":/core/command_panel/modifier_enabled_large.png"));
	toggleStateActionIcon.addFile(QString(":/core/command_panel/modifier_disabled_large.png"), QSize(), QIcon::Normal, QIcon::On);
	toggleModifierStateAction->setIcon(toggleStateActionIcon);
	connect(toggleModifierStateAction, SIGNAL(triggered(bool)), this, SLOT(onModifierToggleState(bool)));
	editToolbar->addAction(toggleModifierStateAction);

	layout->addWidget(splitter);
	layout->addSpacing(4);

	// Create the properties panel.
	propertiesPanel = new PropertiesPanel(NULL);
	propertiesPanel->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
	splitter->addWidget(propertiesPanel);
	splitter->setStretchFactor(1,1);

	connect(&selectionSetListener, SIGNAL(notificationMessage(RefTargetMessage*)), this, SLOT(onSelectionSetMessage(RefTargetMessage*)));
}

/******************************************************************************
* Finds all modifier classes provided by the installed plugins.
******************************************************************************/
void ModifyCommandPage::scanInstalledModifierClasses()
{
	// Create an iterator that retrieves all available modifiers.
	Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(Modifier))) {
		modifierClasses.push_back(clazz);
	}
}

/******************************************************************************
* Resets the modify page to the initial state.
******************************************************************************/
void ModifyCommandPage::reset()
{
	CommandPanelPage::reset();
}

/******************************************************************************
* Is called when the user selects the page.
******************************************************************************/
void ModifyCommandPage::onEnter()
{
	CommandPanelPage::onEnter();
	// Update everything.
	onSelectionChangeComplete(DATASET_MANAGER.currentSelection());
}

/******************************************************************************
* Is called when the user selects another page.
******************************************************************************/
void ModifyCommandPage::onLeave()
{
	CommandPanelPage::onLeave();
	stack->clearStack();
	stack->validate();
	selectionSetListener.setTarget(NULL);
}

/******************************************************************************
* This is called after all changes to the selection set have been completed.
******************************************************************************/
void ModifyCommandPage::onSelectionChangeComplete(SelectionSet* newSelection)
{
	selectionSetListener.setTarget(newSelection);
	updateNodePropertiesEditor();
	stack->validate();
	stack->refreshModifierStack();
}

/******************************************************************************
* This is called by the RefTargetListener that listens to notification messages sent by the
* current selection set.
******************************************************************************/
void ModifyCommandPage::onSelectionSetMessage(RefTargetMessage* msg)
{
	// Update the node properties editor if one of the nodes in the current selection has changed.
	if(msg->type() == NODE_IN_SELECTION_SET_CHANGED) {
		// Check if the sender is a SceneNode because we don't want to update the editor
		// controls each time a sub-object of the selected nodes changes.
		NodeInSelectionSetChangedMessage* msg2 = (NodeInSelectionSetChangedMessage*)msg;
		if(dynamic_object_cast<SceneNode>(msg2->originalMessage()->sender()) != NULL)
			updateNodePropertiesEditor();
	}
}

/******************************************************************************
* Updates the editor controls for the node name, node color etc.
******************************************************************************/
void ModifyCommandPage::updateNodePropertiesEditor()
{
	SelectionSet* sel = static_object_cast<SelectionSet>(selectionSetListener.target());
	CHECK_POINTER(sel);

	nodeNameBox->setEnabled(sel->count() == 1);
	nodeDisplayColorPicker->setEnabled(sel->count() == 1);
	if(sel->empty()) {
        nodeNameBox->setText("No object selected");
        nodeDisplayColorPicker->setColor(Color(palette().color(QPalette::Window)));
	}
	else if(sel->count() == 1) {
		nodeNameBox->setText(sel->node(0)->name());
		nodeDisplayColorPicker->setColor(sel->node(0)->displayColor());
	}
	else {
		nodeNameBox->setText(tr("%n objects selected", NULL, sel->count()));
		nodeDisplayColorPicker->setColor(Color(palette().color(QPalette::Window)));
	}
}

/******************************************************************************
* Is called when the user has entered a new node name in the input text box.
******************************************************************************/
void ModifyCommandPage::onNodeNameEntered()
{
	if(DATASET_MANAGER.currentSelection()->count() != 1)
		return;

	SceneNode* node = DATASET_MANAGER.currentSelection()->node(0);
	CHECK_OBJECT_POINTER(node);

	QString newName = nodeNameBox->text().trimmed();
	if(newName.isEmpty() == false) {
		UNDO_MANAGER.beginCompoundOperation(tr("Rename"));
		node->setName(newName);
		UNDO_MANAGER.endCompoundOperation();
	}

	nodeNameBox->setText(node->name());
	nodeNameBox->selectAll();
}

/******************************************************************************
* Is called when the user has picked a new color for the selected node.
******************************************************************************/
void ModifyCommandPage::onNodeDisplayColorPicked()
{
	if(DATASET_MANAGER.currentSelection()->count() != 1)
		return;

	SceneNode* node = DATASET_MANAGER.currentSelection()->node(0);
	CHECK_OBJECT_POINTER(node);

	UNDO_MANAGER.beginCompoundOperation(tr("Change node color"));
	node->setDisplayColor(nodeDisplayColorPicker->color());
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Is called when the user has selected another item in the modifier stack list box.
******************************************************************************/
void ModifyCommandPage::onModifierStackSelectionChanged()
{
	stack->updatePropertiesPanel();
}

/******************************************************************************
* Is called when the user has selected an item in the modifier class list.
******************************************************************************/
void ModifyCommandPage::onModifierAdd(int index)
{
	if(index >= 0 && stack->isValid()) {
		PluginClassDescriptor* descriptor = (PluginClassDescriptor*)modifierSelector->itemData(index).value<void*>();
		if(descriptor) {
			UNDO_MANAGER.beginCompoundOperation(tr("Apply modifier"));
			try {
				// Create an instance of the modifier...
				Modifier::SmartPtr modifier = static_object_cast<Modifier>(descriptor->createInstance());
				CHECK_OBJECT_POINTER(modifier);
				// .. and apply it.
				stack->applyModifier(modifier.get());
			}
			catch(const Exception& ex) {
				ex.showError();
				UNDO_MANAGER.currentCompoundOperation()->clear();
			}
			UNDO_MANAGER.endCompoundOperation();
			stack->invalidate();
		}
		modifierSelector->setCurrentIndex(0);
	}
}

/******************************************************************************
* Handles the ACTION_MODIFIER_DELETE command.
******************************************************************************/
void ModifyCommandPage::onDeleteModifier()
{
	// Get the selected modifier from the modifier stack box.
	QModelIndexList selection = stackBox->selectionModel()->selectedRows();
	if(selection.empty()) return;
	ModifierStackEntry* selEntry = (ModifierStackEntry*)selection.front().data(Qt::UserRole).value<void*>();
	CHECK_OBJECT_POINTER(selEntry);

	Modifier* modifier = dynamic_object_cast<Modifier>(selEntry->commonObject());
	if(!modifier) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Delete modifier"));
	try {
		// Remove each ModifierApplication from the ModifiedObject it belongs to.
		Q_FOREACH(ModifierApplication* modApp, selEntry->modifierApplications()) {
			OVITO_ASSERT(modApp->modifier() == modifier);
			modApp->modifiedObject()->removeModifier(modApp);
		}
	}
	catch(const Exception& ex) {
		ex.showError();
		UNDO_MANAGER.currentCompoundOperation()->clear();
	}
	UNDO_MANAGER.endCompoundOperation();
	stack->invalidate();
}

/******************************************************************************
* This called when the user double clicks on an item in the modifier stack.
******************************************************************************/
void ModifyCommandPage::onModifierStackDoubleClicked(const QModelIndex& index)
{
	ModifierStackEntry* entry = (ModifierStackEntry*)index.data(Qt::UserRole).value<void*>();
	CHECK_OBJECT_POINTER(entry);

	Modifier* modifier = dynamic_object_cast<Modifier>(entry->commonObject());
	if(modifier) {
		// Toggle enabled state of modifier.
		UNDO_MANAGER.beginCompoundOperation(tr("Toggle modifier state"));
		try {
			modifier->setModifierEnabled(!modifier->isModifierEnabled());
		}
		catch(const Exception& ex) {
			ex.showError();
			UNDO_MANAGER.currentCompoundOperation()->clear();
		}
		UNDO_MANAGER.endCompoundOperation();
	}
}

/******************************************************************************
* Handles the ACTION_MODIFIER_MOVE_UP command, which moves the selected modifier up one entry in the stack.
******************************************************************************/
void ModifyCommandPage::onModifierMoveUp()
{
	// Get the selected modifier from the modifier stack box.
	QModelIndexList selection = stackBox->selectionModel()->selectedRows();
	if(selection.empty()) return;

	ModifierStackEntry* selectedEntry = (ModifierStackEntry*)selection.front().data(Qt::UserRole).value<void*>();
	CHECK_OBJECT_POINTER(selectedEntry);

	if(selectedEntry->modifierApplications().size() != 1) return;

	ModifierApplication::SmartPtr modApp = selectedEntry->modifierApplications()[0];
	ModifiedObject::SmartPtr modObj = modApp->modifiedObject();
	if(modObj == NULL) return;

	OVITO_ASSERT(modObj->modifierApplications().contains(modApp.get()));
	if(modApp == modObj->modifierApplications().back()) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Move modifier up"));
	try {
		// Determine old position in stack.
		int index = modObj->modifierApplications().indexOf(modApp.get());
		// Remove ModifierApplication from the ModifiedObject.
		modObj->removeModifier(modApp.get());
		// Re-insert ModifierApplication into the ModifiedObject.
		modObj->insertModifierApplication(modApp.get(), index+1);
	}
	catch(const Exception& ex) {
		ex.showError();
		UNDO_MANAGER.currentCompoundOperation()->clear();
	}
	UNDO_MANAGER.endCompoundOperation();
	stack->invalidate();
}

/******************************************************************************
* Handles the ACTION_MODIFIER_MOVE_DOWN command, which moves the selected modifier down one entry in the stack.
******************************************************************************/
void ModifyCommandPage::onModifierMoveDown()
{
	// Get the selected modifier from the modifier stack box.
	QModelIndexList selection = stackBox->selectionModel()->selectedRows();
	if(selection.empty()) return;

	ModifierStackEntry* selectedEntry = (ModifierStackEntry*)selection.front().data(Qt::UserRole).value<void*>();
	CHECK_OBJECT_POINTER(selectedEntry);

	if(selectedEntry->modifierApplications().size() != 1) return;

	ModifierApplication::SmartPtr modApp = selectedEntry->modifierApplications()[0];
	ModifiedObject::SmartPtr modObj = modApp->modifiedObject();
	if(modObj == NULL) return;

	OVITO_ASSERT(modObj->modifierApplications().contains(modApp.get()));
	if(modApp == modObj->modifierApplications().front()) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Move modifier down"));
	try {
		// Determine old position in stack.
		int index = modObj->modifierApplications().indexOf(modApp.get());
		// Remove ModifierApplication from the ModifiedObject.
		modObj->removeModifier(modApp.get());
		// Re-insert ModifierApplication into the ModifiedObject.
		modObj->insertModifierApplication(modApp.get(), index-1);
	}
	catch(const Exception& ex) {
		ex.showError();
		UNDO_MANAGER.currentCompoundOperation()->clear();
	}
	UNDO_MANAGER.endCompoundOperation();
	stack->invalidate();
}

/******************************************************************************
* Handles the ACTION_MODIFIER_TOGGLE_STATE command, which toggles the enabled/disable state of the selected modifier.
******************************************************************************/
void ModifyCommandPage::onModifierToggleState(bool newState)
{
	// Get the selected modifier from the modifier stack box.
	QModelIndexList selection = stackBox->selectionModel()->selectedRows();
	if(selection.empty()) return;

	onModifierStackDoubleClicked(selection.front());
}

};
