/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009 Mathias Panzenböck <grosser.meister.morti@gmx.net>
* Copyright (C) 2009-2010 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "SmoothTasks/FixedSizeTaskbarLayout.h"

#include <QApplication>

#include <cmath>

namespace SmoothTasks {

void FixedSizeTaskbarLayout::setFixedCellHeight(qreal fixedCellHeight) {
	if (fixedCellHeight < 1) {
		qWarning("FixedSizeTaskbarLayout::setFixedCellHeight: illegal fixedCellHeight: %g", fixedCellHeight);
		return;
	}

	if (m_fixedCellHeight != fixedCellHeight) {
		m_fixedCellHeight = fixedCellHeight;
		invalidate();
	}
}

int FixedSizeTaskbarLayout::optimumCapacity() const {
	const QRectF effectiveRect(effectiveGeometry());

	const bool isVertical = orientation() == Qt::Vertical;
	const qreal availableWidth = isVertical ? effectiveRect.height() : effectiveRect.width();

	const qreal spacing = this->spacing();
	
	const int itemsPerRow = std::ceil((availableWidth + spacing) / (averagePreferredItemRdWidthStatic(m_cellHeight, true) + spacing));

	return itemsPerRow * maximumRows();
}

int FixedSizeTaskbarLayout::rowOf(const QPointF& pos) const {
	QRectF effectiveRect(effectiveGeometry());
	qreal  spacing = this->spacing();

	if (orientation() == Qt::Vertical) {
		qreal x     = pos.x();
		qreal width = (m_fixedCellHeight + spacing) * m_rows - spacing;

		if (width > effectiveRect.width()) {
			width = effectiveRect.width();
		}

		if (x <= effectiveRect.left()) {
			return 0;
		}
		else if (x >= effectiveRect.right() || width <= 0) {
			return m_rows - 1;
		}
		else {
			return (int) ((x - effectiveRect.left()) * m_rows / width);
		}
	}
	else {
		qreal y      = pos.y();
		qreal height = (m_fixedCellHeight + spacing) * m_rows - spacing;

		if (height > effectiveRect.height()) {
			height = effectiveRect.height();
		}

		if (y <= effectiveRect.top()) {
			return 0;
		}
		else if (y >= effectiveRect.bottom() || height <= 0) {
			return m_rows - 1;
		}
		else {
			return (int) ((y - effectiveRect.top()) * m_rows / height);
		}
	}
}

void FixedSizeTaskbarLayout::updateLayoutStatic() {
	const bool isVertical = orientation() == Qt::Vertical;
	
	const QList<TaskbarItem*>& items = this->items();
	const int N = items.size();
	
	qreal left = 0, top = 0, right = 0, bottom = 0;
	getContentsMargins(&left, &top, &right, &bottom);
	
	const QRectF effectiveRect(effectiveGeometry());
	const qreal availableWidth  = isVertical ? effectiveRect.height() : effectiveRect.width();
	const qreal availableHeight = isVertical ? effectiveRect.width()  : effectiveRect.height();
	const qreal spacing         = this->spacing();
	
#define CELL_HEIGHT(ROWS) (((availableHeight + spacing) / ((qreal) (ROWS))) - spacing)
	
	// if there is nothing to layout fill in some dummy data and leave
	if (N == 0) {
		m_rows = minimumRows();
		m_cellHeight = qMin(m_fixedCellHeight, CELL_HEIGHT(m_rows));
		m_rowInfos.clear();
		return;
	}
	
	int itemsPerRow = N;
	int rows = maximumRows();
	
	itemsPerRow = std::ceil((availableWidth + spacing) / (averagePreferredItemRdWidthStatic(qMin(m_fixedCellHeight, CELL_HEIGHT(rows))) + spacing));
	
	if (itemsPerRow * rows < N) {
		itemsPerRow = std::ceil(((qreal) N) / rows);
	}
	
	QList<RowInfo> rowInfos;
	
	buildRows(itemsPerRow, rowInfos, rows);
	
	m_rows = rows;
	m_cellHeight = qMin(m_fixedCellHeight, CELL_HEIGHT(m_rows));
	m_rowInfos = rowInfos;
	updateItemsRowCache();
	
#undef CELL_HEIGHT
}

} // namespace SmoothTasks
