/* Tic-Tac-Toe plasmoid
 * Copyright (C) 2009 Mauro E. Bender <mbender@dc.uba.ar>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "tictactoe.h"

Tictactoe::Tictactoe(std::string j1, std::string j2){
	_players[0] = j1;
	_players[1] = j2;
	
	_currentPlayer = 1;
	_status = INGAME;
	
	for(int i = 0; i < 3; i++){
		for(int j = 0; j < 3; j++){
			_grid[i][j] = 0;
		}
	}
}

Tictactoe::Tictactoe( const Tictactoe& t){
	*this = t;
}

int Tictactoe::getValue(int f, int c) const{
	return _grid[f][c];
}

void Tictactoe::setValue(int f, int c, int v){
	if( v < 0 || 2 < v)
		return;
	
	_grid[f][c] = v;
}

void Tictactoe::setPlayerName( int j, std::string name){
	if(j < 1 || j > 2)
		return;
	
	_players[j-1] = name;
}

std::string Tictactoe::getPlayerName( int j ) const{
	return _players[j-1];
}

std::string Tictactoe::currentPlayer() const{
	return _players[ _currentPlayer - 1 ];
}

void Tictactoe::touch(int f, int c){
	if( _status != INGAME || _grid[f][c] != 0)
		return;
	
	_grid[f][c] = _currentPlayer;
	checkWinner();
	
	if( _status != INGAME )
		return;
	
	if (_currentPlayer == 1)
		_currentPlayer = 2;
	else
		_currentPlayer = 1;
}

Tictactoe& Tictactoe::operator=( const Tictactoe& t ){
	_currentPlayer = t._currentPlayer;
	
	_players[0] = t._players[0];
	_players[1] = t._players[1];
	
	for( int i = 0; i < 3; i++){
		for(int j = 0; j < 3; j++){
			_grid[i][j] = t._grid[i][j];
		}
	}
	
	return *this;
}

bool Tictactoe::hasWinner(){
	return _status == WINNER;
}

std::string Tictactoe::winner(){
	return _players[_currentPlayer-1];
}

void Tictactoe::reset(){
	_status = INGAME;
	_currentPlayer = 1;
	
	for(int i = 0; i < 3; i++){
		for(int j = 0; j < 3; j++){
			_grid[i][j] = 0;
		}
	}
}

GameStatus Tictactoe::gameStatus(){
	return _status;
}

void Tictactoe::checkWinner(){
	//We check if there is a winner
	bool hasWinner = false;
	
	for (int i = 0; i < 3; i++){
		hasWinner |= _grid[i][0] != 0 && _grid[i][0] == _grid[i][1] && _grid[i][0] == _grid[i][2] && _grid[i][1] == _grid[i][2];
	}
	
	for (int i = 0; i < 3; i++){
		hasWinner |= _grid[0][i] != 0 && _grid[0][i] == _grid[1][i] && _grid[0][i] == _grid[2][i] && _grid[1][i] == _grid[2][i];
	}
	
	hasWinner |= _grid[0][0] != 0 && _grid[0][0] == _grid[1][1] && _grid[0][0] == _grid[2][2] && _grid[1][1] == _grid[2][2];
	hasWinner |= _grid[2][0] != 0 && _grid[2][0] == _grid[1][1] && _grid[2][0] == _grid[0][2] && _grid[1][1] == _grid[0][2];
	
	if( hasWinner ){
		_status = WINNER;
		return;
	}
	
	//We check if there is a tie
	bool tie = true;
	
	for(int i = 0; i < 3; i++){
		for(int j = 0; j < 3; j++){
			tie &= _grid[i][j] != 0;
		}
	}
	
	if( tie)
		_status = TIE;
}
