
/*****************************************************************************
*
* Copyright (c) 2003-2017 by The University of Queensland
* http://www.uq.edu.au
*
* Primary Business: Queensland, Australia
* Licensed under the Apache License, version 2.0
* http://www.apache.org/licenses/LICENSE-2.0
*
* Development until 2012 by Earth Systems Science Computational Center (ESSCC)
* Development 2012-2013 by School of Earth Sciences
* Development from 2014 by Centre for Geoscience Computing (GeoComp)
*
*****************************************************************************/

#ifndef __WEIPA_RIPLEYDOMAIN_H__
#define __WEIPA_RIPLEYDOMAIN_H__

#include <weipa/DomainChunk.h>
#include <weipa/RipleyElements.h>
#include <boost/enable_shared_from_this.hpp>

namespace weipa {

/// \brief Represents a full Ripley domain including nodes and elements.
///
/// This class represents a Ripley domain including nodes, cells, and face
/// elements. It provides functionality to read a domain from a file
/// (generated by the domain's dump() method) or directly through an instance
/// of RipleyDomain.
///
/// Once initialised, the domain can be saved to a Silo file or its nodes
/// and elements accessed through the respective methods.
class RipleyDomain : public DomainChunk, public boost::enable_shared_from_this<RipleyDomain>
{
public:
    RipleyDomain();
    RipleyDomain(const RipleyDomain& m);
    virtual ~RipleyDomain() {}
    virtual bool initFromEscript(const escript::AbstractDomain* domain);
    virtual bool initFromFile(const std::string& filename);
    virtual bool writeToSilo(DBfile* dbfile, const std::string& pathInSilo,
                             const StringVec& labels, const StringVec& units,
                             bool writeMeshData);
    virtual void reorderGhostZones(int ownIndex);
    virtual void removeGhostZones(int ownIndex);
    virtual StringVec getMeshNames() const;
    virtual StringVec getVarNames() const;
    virtual ElementData_ptr getElementsByName(const std::string& name) const;
    virtual NodeData_ptr getMeshByName(const std::string& name) const;
    virtual DataVar_ptr getDataVarByName(const std::string& name) const;
    virtual Centering getCenteringForFunctionSpace(int fsCode) const;
    virtual NodeData_ptr getMeshForFunctionSpace(int fsCode) const;
    virtual ElementData_ptr getElementsForFunctionSpace(int fsCode) const;
    virtual NodeData_ptr getNodes() const { return nodes; }
    virtual std::string getSiloPath() const { return siloPath; }
    virtual void setSiloPath(const std::string& path)  { siloPath = path; }

private:
    bool initialized;
    RipleyNodes_ptr    nodes;
    RipleyElements_ptr cells;
    RipleyElements_ptr faces;
    std::string        siloPath;
};

} // namespace weipa

#endif // __WEIPA_RIPLEYDOMAIN_H__

