% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unsplitAltExps.R
\name{unsplitAltExps}
\alias{unsplitAltExps}
\title{Unsplit the alternative experiments}
\usage{
unsplitAltExps(sce, prefix.rows = TRUE, prefix.cols = TRUE, delayed = TRUE)
}
\arguments{
\item{sce}{A \linkS4class{SingleCellExperiment} containing alternative experiments in the \code{\link{altExps}} slot.}

\item{prefix.rows}{Logical scalar indicating whether the (non-\code{NULL}) row names should be prefixed with the name of the alternative experiment.}

\item{prefix.cols}{Logical scalar indicating whether the names of column-related fields should be prefixed with the name of the alternative experiment.
If \code{NA}, any \code{\link{colData}} of the \code{\link{altExps}} are ignored.}

\item{delayed}{Logical scalar indicating whether the combining of the assays should be delayed.}
}
\value{
A SingleCellExperiment where all features in the alternative experiments of \code{sce} are now features in the main experiment.
The output object has no alternative experiments of its own.
}
\description{
Combine the main and alternative experiments back into one \linkS4class{SingleCellExperiment} object.
This is effectively the reverse operation to \code{\link{splitAltExps}}.
}
\details{
This function is intended for downstream applications that accept a SingleCellExperiment but are not aware of the \code{\link{altExps}} concept.
By consolidating all data together, applications like \pkg{iSEE} can use the same machinery to visualize any feature of interest across all modalities.
However, for quantitative analyses, it is usually preferable to keep different modalities separate.

Assays with the same name are \code{\link{rbind}}ed together in the output object.
If a particular name is not present for any experiment, its values are filled in with the appropriately typed \code{NA} instead.
By default, this is done efficiently via \linkS4class{ConstantMatrix} abstractions to avoid actually creating a dense matrix of \code{NA}s.
If \code{delayed=FALSE}, the combining of matrices is done without any \linkS4class{DelayedArray} wrappers,
yielding a simpler matrix representation at the cost of increasing memory usage.

Any \code{\link{colData}} or \code{\link{reducedDims}} in the alternative experiments are added to those of the main experiment.
The names of these migrated fields are prefixed by the name of the alternative experiment if \code{prefix.cols=TRUE}.

Setting \code{prefix.rows=FALSE}, \code{prefix.cols=NA} and \code{delayed=FALSE} will reverse the effects of \code{\link{splitAltExps}}.
}
\examples{
counts <- matrix(rpois(10000, 5), ncol=100)
sce <- SingleCellExperiment(assays=list(counts=counts))
feat.type <- sample(c("endog", "ERCC", "adt"), nrow(sce),
    replace=TRUE, p=c(0.8, 0.1, 0.1))
sce <- splitAltExps(sce, feat.type)

# Making life a little more complicated.
logcounts(sce) <- log2(counts(sce) + 1)
sce$cluster <- sample(5, ncol(sce), replace=TRUE)
reducedDim(sce, "PCA") <- matrix(rnorm(ncol(sce)*2), ncol=2)
 
# Now, putting Humpty Dumpty back together again. 
restored <- unsplitAltExps(sce)
restored

}
\seealso{
\code{\link{splitAltExps}}, which does the reverse operation of this function.
}
\author{
Aaron Lun
}
