// SPDX-License-Identifier: BSD-3-Clause
//
// Copyright 2014 Raritan Inc. All rights reserved.

package com.raritan.json_rpc.examples;

import java.util.List;
import java.util.Scanner;

import com.raritan.idl.TypeInfo;
import com.raritan.idl.pdumodel.Inlet;
import com.raritan.idl.pdumodel.Inlet_2_0_0;
import com.raritan.idl.pdumodel.Outlet;
import com.raritan.idl.pdumodel.Outlet_2_0_0;
import com.raritan.idl.pdumodel.Pdu;
import com.raritan.idl.pdumodel.Pdu_2_0_0;
import com.raritan.idl.pdumodel.Pdu_3_0_0;
import com.raritan.idl.pdumodel.Pdu_4_0_0;
import com.raritan.idl.sensors.NumericSensor;
import com.raritan.idl.sensors.NumericSensor_3_0_0;
import com.raritan.idl.sensors.NumericSensor_4_0_0;
import com.raritan.idl.sensors.Sensor_4_0_0;
import com.raritan.idl.session.SessionManager;
import com.raritan.idl.session.SessionManager_2_0_0;
import com.raritan.json_rpc.Agent;
import com.raritan.json_rpc.ObjectProxy;
import com.raritan.json_rpc.pdumodel.Inlet_Proxy;
import com.raritan.json_rpc.pdumodel.Inlet_2_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Outlet_Proxy;
import com.raritan.json_rpc.pdumodel.Outlet_2_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Pdu_2_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Pdu_3_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Pdu_4_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Pdu_Proxy;
import com.raritan.json_rpc.sensors.NumericSensor_3_0_0_Proxy;
import com.raritan.json_rpc.sensors.NumericSensor_4_0_0_Proxy;
import com.raritan.json_rpc.sensors.NumericSensor_Proxy;
import com.raritan.json_rpc.session.SessionManager_Proxy;
import com.raritan.json_rpc.session.SessionManager_2_0_0_Proxy;

public class DumpPdu {

    private static SessionManager sessmanV1 = null;
    private static SessionManager_2_0_0 sessmanV2 = null;

    private static String createSession(Agent agent) throws Exception {
	ObjectProxy sessmanProxy = new ObjectProxy(agent, "/session");
	TypeInfo typeInfo = sessmanProxy.getTypeInfo();

	if (typeInfo.isCallCompatible(SessionManager.typeInfo)) {
	    // SessionManager_1_0_0: Authentication token is part of the Session
	    //                       structure returned by newSession
	    sessmanV1 = SessionManager_Proxy.staticCast(sessmanProxy);
	    SessionManager.NewSessionResult result = sessmanV1.newSession();
	    if (result._ret_ == 0) {
		return result.session.token;
	    } else {
		System.out.println("ERROR: Failed to create a session with SessionManager_1_0_0");
	    }
	} else if (typeInfo.isCallCompatible(SessionManager_2_0_0.typeInfo)) {
	    // SessionManager_2_0_0: Authentication token is not part of the Session
	    //                       structure; it is returned as an out-parameter
	    sessmanV2 = SessionManager_2_0_0_Proxy.staticCast(sessmanProxy);
	    SessionManager_2_0_0.NewSessionResult result = sessmanV2.newSession();
	    if (result._ret_ == 0) {
		return result.token;
	    } else {
		System.out.println("ERROR: Failed to create a session with SessionManager_2_0_0");
	    }
	} else {
	    System.out.println("Unsupported version of SessionManager interface: " + typeInfo);
	}
	return null;
    }

    private static void closeSession() throws Exception {
	if (sessmanV1 != null) {
	    sessmanV1.closeCurrentSession(SessionManager.CloseReason.CLOSE_REASON_LOGOUT);
	} else if (sessmanV2 != null) {
	    sessmanV2.closeCurrentSession(SessionManager_2_0_0.CloseReason.CLOSE_REASON_LOGOUT);
	}
    }

    private static String formatSensorReading(double reading, int decdigits, String unit) {
	String result = String.format("%." + decdigits + "f", reading);
	if (!unit.isEmpty()) {
	    result += " " + unit;
	}
	return result;
    }

    private static void dumpNumericSensorV1(NumericSensor nsensor, String label) throws Exception {
	NumericSensor.MetaData metadata = nsensor.getMetaData()._ret_;
	NumericSensor.Reading reading = nsensor.getReading()._ret_;

	String unit;
	switch (metadata.type.unit) {
	    case NONE:          unit = ""; break;
	    case VOLT:          unit = "V"; break;
	    case AMPERE:        unit = "A"; break;
	    case WATT:          unit = "W"; break;
	    case VOLT_AMP:      unit = "VA"; break;
	    case WATT_HOUR:     unit = "Wh"; break;
	    case VOLT_AMP_HOUR: unit = "VAh"; break;
	    case HZ:            unit = "Hz"; break;
	    default:            unit = "?"; break;
	}

	if (reading.valid) {
	    System.out.println(label + formatSensorReading(reading.value, metadata.decdigits, unit));
	} else if (reading.available) {
	    System.out.println(label + "no reading");
	} else {
	    System.out.println(label + "unavailable");
	}
    }

    private static void dumpNumericSensorV3(NumericSensor_3_0_0 nsensor, String label) throws Exception {
	// The main difference between NumericSensor interface versions 1 and 3
	// was the addition of new sensor types and units. Therefore this code is
	// identical to dumpNumericSensorV1 except for the use of more recent
	// data types.

	NumericSensor_3_0_0.MetaData metadata = nsensor.getMetaData()._ret_;
	NumericSensor_3_0_0.Reading reading = nsensor.getReading()._ret_;

	String unit;
	switch (metadata.type.unit) {
	    case NONE:          unit = ""; break;
	    case VOLT:          unit = "V"; break;
	    case AMPERE:        unit = "A"; break;
	    case WATT:          unit = "W"; break;
	    case VOLT_AMP:      unit = "VA"; break;
	    case WATT_HOUR:     unit = "Wh"; break;
	    case VOLT_AMP_HOUR: unit = "VAh"; break;
	    case HZ:            unit = "Hz"; break;
	    default:            unit = "?"; break;
	}

	if (reading.valid) {
	    System.out.println(label + formatSensorReading(reading.value, metadata.decdigits, unit));
	} else if (reading.available) {
	    System.out.println(label + "no reading");
	} else {
	    System.out.println(label + "unavailable");
	}
    }

    private static void dumpNumericSensorV4(NumericSensor_4_0_0 nsensor, String label) throws Exception {
	// Starting with NumericSensor_4_0_0 the existing ReadingType, Type and Unit
	// enumerations have been replaced with plain integers so that new sensor
	// types can now be implemented without formally breaking the interface
	// and incrementing the major version.

	NumericSensor_4_0_0.MetaData metadata = nsensor.getMetaData()._ret_;
	NumericSensor_4_0_0.Reading reading = nsensor.getReading()._ret_;

	String unit;
	switch (metadata.type.unit) {
	    case Sensor_4_0_0.NONE:          unit = ""; break;
	    case Sensor_4_0_0.VOLT:          unit = "V"; break;
	    case Sensor_4_0_0.AMPERE:        unit = "A"; break;
	    case Sensor_4_0_0.WATT:          unit = "W"; break;
	    case Sensor_4_0_0.VOLT_AMP:      unit = "VA"; break;
	    case Sensor_4_0_0.WATT_HOUR:     unit = "Wh"; break;
	    case Sensor_4_0_0.VOLT_AMP_HOUR: unit = "VAh"; break;
	    case Sensor_4_0_0.HZ:            unit = "Hz"; break;
	    default:                         unit = "?"; break;
	}

	if (reading.valid) {
	    System.out.println(label + formatSensorReading(reading.value, metadata.decdigits, unit));
	} else if (reading.available) {
	    System.out.println(label + "no reading");
	} else {
	    System.out.println(label + "unavailable");
	}
    }

    private static void dumpNumericSensor(ObjectProxy proxy, String label) throws Exception {
	if (proxy == null) {
	    return;
	}

	// There are four incompatible versions of the NumericSensor
	// interface. NumericSensor_2_x_y was only used in EMX, so we
	// don't need to support it here.

	TypeInfo typeInfo = proxy.getTypeInfo();
	if (typeInfo.isCallCompatible(NumericSensor.typeInfo)) {
	    dumpNumericSensorV1(NumericSensor_Proxy.staticCast(proxy), label);
	} else if (typeInfo.isCallCompatible(NumericSensor_3_0_0.typeInfo)) {
	    dumpNumericSensorV3(NumericSensor_3_0_0_Proxy.staticCast(proxy), label);
	} else if (typeInfo.isCallCompatible(NumericSensor_4_0_0.typeInfo)) {
	    dumpNumericSensorV4(NumericSensor_4_0_0_Proxy.staticCast(proxy), label);
	} else {
	    System.out.println("Unsupported version of NumericSensor interface: " + typeInfo);
	}
    }

    private static void dumpInlet(ObjectProxy proxy) throws Exception {
	TypeInfo typeInfo = proxy.getTypeInfo();
	if (typeInfo.isCallCompatible(Inlet.typeInfo)) {
	    dumpInletV1(Inlet_Proxy.staticCast(proxy));
	} else if (typeInfo.isCallCompatible(Inlet_2_0_0.typeInfo)){
	    dumpInletV2(Inlet_2_0_0_Proxy.staticCast(proxy));
	} else {
	    System.out.println("Unsupported version of Inlet interface: " + typeInfo);
	}
    }

    private static void dumpInletV1(Inlet inlet) throws Exception {
	Inlet.MetaData metadata = inlet.getMetaData()._ret_;
	Inlet.Settings settings = inlet.getSettings()._ret_;
	Inlet.Sensors sensors = inlet.getSensors()._ret_;

	String name = settings.name;
	if (name.isEmpty()) name = "no name";
	System.out.printf("Inlet %s (%s):%n", metadata.label, name);

	System.out.println("  Plug type:      " + metadata.plugType);
	dumpNumericSensor((ObjectProxy)sensors.voltage,       "  Voltage:        ");
	dumpNumericSensor((ObjectProxy)sensors.current,       "  Current:        ");
	dumpNumericSensor((ObjectProxy)sensors.activePower,   "  Active Power:   ");
	dumpNumericSensor((ObjectProxy)sensors.apparentPower, "  Apparent Power: ");
	dumpNumericSensor((ObjectProxy)sensors.powerFactor,   "  Power Factor:   ");
	System.out.println();
    }

    private static void dumpInletV2(Inlet_2_0_0 inlet) throws Exception {
	Inlet_2_0_0.MetaData metadata = inlet.getMetaData()._ret_;
	Inlet_2_0_0.Settings settings = inlet.getSettings()._ret_;
	Inlet_2_0_0.Sensors sensors = inlet.getSensors()._ret_;

	String name = settings.name;
	if (name.isEmpty()) name = "no name";
	System.out.printf("Inlet %s (%s):%n", metadata.label, name);

	System.out.println("  Plug type:      " + metadata.plugType);
	dumpNumericSensor((ObjectProxy)sensors.voltage,       "  Voltage:        ");
	dumpNumericSensor((ObjectProxy)sensors.current,       "  Current:        ");
	dumpNumericSensor((ObjectProxy)sensors.activePower,   "  Active Power:   ");
	dumpNumericSensor((ObjectProxy)sensors.apparentPower, "  Apparent Power: ");
	dumpNumericSensor((ObjectProxy)sensors.powerFactor,   "  Power Factor:   ");
	System.out.println();
    }

    private static void dumpOutlet(ObjectProxy proxy) throws Exception {
	TypeInfo typeInfo = proxy.getTypeInfo();
	if (typeInfo.isCallCompatible(Outlet.typeInfo)) {
	    dumpOutletV1(Outlet_Proxy.staticCast(proxy));
	} else if (typeInfo.isCallCompatible(Outlet_2_0_0.typeInfo)){
	    dumpOutletV2(Outlet_2_0_0_Proxy.staticCast(proxy));
	} else {
	    System.out.println("Unsupported version of Outlet interface: " + typeInfo);
	}
    }

    private static void dumpOutletV1(Outlet outlet) throws Exception {
	Outlet.MetaData metadata = outlet.getMetaData()._ret_;
	Outlet.Settings settings = outlet.getSettings()._ret_;
	Outlet.Sensors sensors = outlet.getSensors()._ret_;

	String name = settings.name;
	if (name.isEmpty()) name = "no name";
	System.out.printf("Outlet %s (%s):%n", metadata.label, name);

	System.out.println("  Receptacle type: " + metadata.receptacleType);
	dumpNumericSensor((ObjectProxy)sensors.voltage,       "  Voltage:         ");
	dumpNumericSensor((ObjectProxy)sensors.current,       "  Current:         ");
	dumpNumericSensor((ObjectProxy)sensors.activePower,   "  Active Power:    ");
	dumpNumericSensor((ObjectProxy)sensors.apparentPower, "  Apparent Power:  ");
	dumpNumericSensor((ObjectProxy)sensors.powerFactor,   "  Power Factor:    ");
	System.out.println();
    }

    private static void dumpOutletV2(Outlet_2_0_0 outlet) throws Exception {
	Outlet_2_0_0.MetaData metadata = outlet.getMetaData()._ret_;
	Outlet_2_0_0.Settings settings = outlet.getSettings()._ret_;
	Outlet_2_0_0.Sensors sensors = outlet.getSensors()._ret_;

	String name = settings.name;
	if (name.isEmpty()) name = "no name";
	System.out.printf("Outlet %s (%s):%n", metadata.label, name);

	System.out.println("  Receptacle type: " + metadata.receptacleType);
	dumpNumericSensor((ObjectProxy)sensors.voltage,       "  Voltage:         ");
	dumpNumericSensor((ObjectProxy)sensors.current,       "  Current:         ");
	dumpNumericSensor((ObjectProxy)sensors.activePower,   "  Active Power:    ");
	dumpNumericSensor((ObjectProxy)sensors.apparentPower, "  Apparent Power:  ");
	dumpNumericSensor((ObjectProxy)sensors.powerFactor,   "  Power Factor:    ");
	System.out.println();
    }

    private static void dumpPduV1(Pdu pdu) throws Exception {
	Pdu.MetaData metadata = pdu.getMetaData()._ret_;
	System.out.println("Model:            " + metadata.nameplate.model);
	System.out.println("Serial number:    "  + metadata.nameplate.serialNumber);
	System.out.printf ("Rating:           %s, %s, %s, %s%n",
		metadata.nameplate.rating.voltage, metadata.nameplate.rating.frequency,
		metadata.nameplate.rating.current, metadata.nameplate.rating.power);
	System.out.println("Firmware version: " + metadata.fwRevision);

	Pdu.Settings settings = pdu.getSettings()._ret_;
	System.out.println("PDU name:         " + settings.name);
	System.out.println();

	List<?> inlets = pdu.getInlets()._ret_;
	for (Object inlet : inlets) {
	    dumpInlet((ObjectProxy)inlet);
	}

	List<?> outlets = pdu.getOutlets()._ret_;
	for (Object outlet : outlets) {
	    dumpOutlet((ObjectProxy)outlet);
	}
    }

    private static void dumpPduV2(Pdu_2_0_0 pdu) throws Exception {
	Pdu_2_0_0.MetaData metadata = pdu.getMetaData()._ret_;
	System.out.println("Model:            " + metadata.nameplate.model);
	System.out.println("Serial number:    "  + metadata.nameplate.serialNumber);
	System.out.printf ("Rating:           %s, %s, %s, %s%n",
		metadata.nameplate.rating.voltage, metadata.nameplate.rating.frequency,
		metadata.nameplate.rating.current, metadata.nameplate.rating.power);
	System.out.println("Firmware version: " + metadata.fwRevision);

	Pdu_2_0_0.Settings settings = pdu.getSettings()._ret_;
	System.out.println("PDU name:         " + settings.name);
	System.out.println();

	// We only checked whether the proxy object was call-compatible with
	// the Pdu_2_0_0 interface, so we can't rely on the version numbers
	// of the returned inlets and outlets

	List<?> inlets = pdu.getInlets()._ret_;
	for (Object inlet : inlets) {
	    dumpInlet((ObjectProxy)inlet);
	}

	List<?> outlets = pdu.getOutlets()._ret_;
	for (Object outlet : outlets) {
	    dumpOutlet((ObjectProxy)outlet);
	}
    }

    private static void dumpPduV3(Pdu_3_0_0 pdu) throws Exception {
	Pdu_3_0_0.MetaData metadata = pdu.getMetaData()._ret_;
	System.out.println("Model:            " + metadata.nameplate.model);
	System.out.println("Serial number:    "  + metadata.nameplate.serialNumber);
	System.out.printf ("Rating:           %s, %s, %s, %s%n",
		metadata.nameplate.rating.voltage, metadata.nameplate.rating.frequency,
		metadata.nameplate.rating.current, metadata.nameplate.rating.power);
	System.out.println("Firmware version: " + metadata.fwRevision);

	Pdu_3_0_0.Settings settings = pdu.getSettings()._ret_;
	System.out.println("PDU name:         " + settings.name);
	System.out.println();

	// We only checked whether the proxy object was call-compatible with
	// the Pdu_3_0_0 interface, so we can't rely on the version numbers
	// of the returned inlets and outlets

	List<?> inlets = pdu.getInlets()._ret_;
	for (Object inlet : inlets) {
	    dumpInlet((ObjectProxy)inlet);
	}

	List<?> outlets = pdu.getOutlets()._ret_;
	for (Object outlet : outlets) {
	    dumpOutlet((ObjectProxy)outlet);
	}
    }

    private static void dumpPduV4(Pdu_4_0_0 pdu) throws Exception {
	Pdu_4_0_0.MetaData metadata = pdu.getMetaData()._ret_;
	System.out.println("Model:            " + metadata.nameplate.model);
	System.out.println("Serial number:    "  + metadata.nameplate.serialNumber);
	System.out.printf ("Rating:           %s, %s, %s, %s%n",
		metadata.nameplate.rating.voltage, metadata.nameplate.rating.frequency,
		metadata.nameplate.rating.current, metadata.nameplate.rating.power);
	System.out.println("Firmware version: " + metadata.fwRevision);

	Pdu_4_0_0.Settings settings = pdu.getSettings()._ret_;
	System.out.println("PDU name:         " + settings.name);
	System.out.println();

	// We only checked whether the proxy object was call-compatible with
	// the Pdu_4_0_0 interface, so we can't rely on the version numbers
	// of the returned inlets and outlets

	List<?> inlets = pdu.getInlets()._ret_;
	for (Object inlet : inlets) {
	    dumpInlet((ObjectProxy)inlet);
	}

	List<?> outlets = pdu.getOutlets()._ret_;
	for (Object outlet : outlets) {
	    dumpOutlet((ObjectProxy)outlet);
	}
    }

    private static void dumpPdu(ObjectProxy proxy) throws Exception {
	// Check which Pdu object we have -- there
	// have been incompatible changes between those versions.

	TypeInfo typeInfo = proxy.getTypeInfo();
	if (typeInfo.isCallCompatible(Pdu.typeInfo)) {
	    dumpPduV1(Pdu_Proxy.staticCast(proxy));
	} else if (typeInfo.isCallCompatible(Pdu_2_0_0.typeInfo)) {
	    dumpPduV2(Pdu_2_0_0_Proxy.staticCast(proxy));
	} else if (typeInfo.isCallCompatible(Pdu_3_0_0.typeInfo)) {
	    dumpPduV3(Pdu_3_0_0_Proxy.staticCast(proxy));
	} else if (typeInfo.isCallCompatible(Pdu_4_0_0.typeInfo)) {
	    dumpPduV4(Pdu_4_0_0_Proxy.staticCast(proxy));
	} else {
	    System.out.println("Unsupported version of Pdu interface: " + typeInfo);
	}
    }

    public static void main(String[] args) {
	String host;
	if (args.length >= 1) {
	    host = args[0];
	} else {
	    System.out.println("Please enter PDU URL (default: https://10.0.42.2):");
	    Scanner input = new Scanner(System.in);
	    host = input.nextLine();
	    input.close();
	}
	if (host.isEmpty()) {
	    host = "https://10.0.42.2";
	}
	if (!host.contains("://")) {
	    host = "https://" + host;
	}
	System.out.printf("Connecting to %s ...%n", host);

	try {
	    // Create an agent for performing JSON-RPC objects over HTTP(S)
	    Agent agent = new Agent(host);
	    agent.setUsername("admin");
	    agent.setPassword("raritan");

	    // Optional: Create an authenticated session; it's also possible
	    // to authenticate all requests with username and password.
	    String token = createSession(agent);
	    if (token != null) {
		// Session successfully created, authenticate all subsequent
		// requests with its secret token
		agent.setToken(token);
	    }

	    // Create the root PDU object and dump its topology
	    ObjectProxy pduProxy = new ObjectProxy(agent, "/model/pdu/0");
	    dumpPdu(pduProxy);

	    // Finally: close the authenticated session
	    closeSession();
	} catch (Exception e) {
	    e.printStackTrace();
	}
    }

}
