///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// OUT ML: Matrix and Vector Output
//	      in a MatLab-like format
// 
// author: Pierre.Saramito@imag.fr
//
// date: 21 january 1997
//
// note: code inspired from octave-1.1.1
//	 mail:jwe@che.utexas.edu (John W. Eaton)
//	 ftp://ftp.che.utexas.edu:/pub/octave
//
# include "rheolef/csr.h"
# include "rheolef/outml.h"
# include "rheolef/vec.h"
# include "rheolef/avec.h"
# include "rheolef/iotrick.h" // TODO: delete it !
using namespace rheolef;
using namespace std;
namespace rheolef { 

// global var
ml_preferences ml_pref;

// circumvent bug in shared libraries :
// ml_pref cstor not called : force initialization
static void init_ml_pref() {
    static bool already_init = false;
    if (!already_init) {
        ml_pref.reset();
        already_init = true;
    }
}
int screenwidth = 80; // TODO: could use readline.
int
terminal_columns (void)
{
	extern int screenwidth;
	return screenwidth > 0 ? screenwidth : 80;
}
// Current format string for floats numbers
static char *curr_fmt = 0;

template <class T>
static
void
set_format (ostream& os, const T& a, int& fw)
{
        init_ml_pref();
	curr_fmt = 0;
  	if (ml_pref.free_format) return;

  	static char fmt_buf[128];
	int sign       = a.any_element_is_negative ();
	int inf_or_nan = a.any_element_is_inf_or_nan ();
	Float max_abs_  = a.max_abs ();
	Float min_abs_  = a.min_abs ();
	int x_max = ((max_abs_ == Float(0)) ? 0 : ((int) floor (log10 (max_abs_) + 1.0)));
	int x_min = ((min_abs_ == Float(0)) ? 0 : ((int) floor (log10 (min_abs_) + 1.0)));

	int prec = os.precision();
	if (os.width() != 0)
	    ml_pref.output_max_field_width = os.width();
	else
	    ml_pref.output_max_field_width = screenwidth;

	bool has_fixed = ((cout.flags() & ios::floatfield) == ios::fixed);
	bool has_scien = ((cout.flags() & ios::floatfield) == ios::scientific);
        if (!has_fixed && !has_scien) {
	    ml_pref.bank_format = false;
            ml_pref.print_e     = false;
        } else if (has_scien) {
	    ml_pref.bank_format = false;
            ml_pref.print_e     = true;
        } else if (has_fixed) {
	    ml_pref.bank_format = true;
            ml_pref.print_e     = false;
	}
	int ld, rd;

  	if (ml_pref.bank_format) {
      
	    int digits = x_max > x_min ? x_max : x_min;
	    fw = digits <= 0 ? 4 : digits + 3;
      	    if (inf_or_nan && fw < 3)
		fw = 3;
      	    fw += sign;
      	    rd = 2;

    	} else if (a.all_elements_are_int_or_inf_or_nan ()) {
      
	    int digits = x_max > x_min ? x_max : x_min;
      	    fw = digits <= 0 ? 1 : digits;
      	    if (inf_or_nan && fw < 3)
		fw = 3;
      	    fw += sign;
      	    rd = 0;
    	} else {
	    int ld_max, rd_max;
      	    if (x_max > 0) {
		ld_max = x_max;
	  	rd_max = prec - x_max;
	  	x_max++;
	    } else {
	        ld_max = 1;
	        rd_max = prec - x_max;
	        x_max = -x_max + 1;
	    }
	    int ld_min, rd_min;
      	    if (x_min > 0) {
	        ld_min = x_min;
	        rd_min = prec - x_min;
	        x_min++;
	    } else {
	        ld_min = 1;
	        rd_min = prec - x_min;
	        x_min = -x_min + 1;
	    }
	    ld = ld_max > ld_min ? ld_max : ld_min;
            rd = rd_max > rd_min ? rd_max : rd_min;
	    fw = ld + 1 + rd;
      	    if (inf_or_nan && fw < 3)
	        fw = 3;
      	    fw += sign;
        }
	if (! ml_pref.bank_format && (fw > ml_pref.output_max_field_width || ml_pref.print_e)) {
      
	    int exp_field = 4;
      	    if (x_max > 100 || x_min > 100)
		exp_field++;
	    fw = 2 + prec + exp_field;
      	    if (inf_or_nan && fw < 3)
		fw = 3;
      	    fw += sign;
	    if (ml_pref.print_big_e)
		sprintf (fmt_buf, "%%%d.%dE", fw, prec - 1);
      	    else
		sprintf (fmt_buf, "%%%d.%de", fw, prec - 1);
    	} else {
      
	    	sprintf (fmt_buf, "%%%d.%df", fw, rd);
    	}
	curr_fmt = &fmt_buf[0];
}
static
inline
void
pr_any_float (const char *fmt, ostream& os, Float d, int fw = 0)
{
	if (d == Float(-0.)) d = 0;
  	if (fmt) {
      	    if (xisinf (d)) {
	        const char *s;
	        if (d < Float(0.))
	            s = "-Inf";
	        else
	            s = "Inf";

	        if (fw > 0) {
	            skit_form2(os, "%*s", fw, s);
	        } else {
	            os << s;
		}
	    } else if (xisnan (d)) {
	        if (fw > 0) {
	            skit_form2(os, "%*s", fw, "NaN");
	        } else {
	            os << "NaN";
		}
	    } else {
	            skit_form(os, fmt, double(d));
	    }
        } else {
    	            os << double(d);
        }
}
static inline void
pr_float (ostream& os, Float d, int fw = 0)
{
  	pr_any_float (curr_fmt, os, d, fw);
}
static void
print_empty_vector (ostream& os, int n)
{
    init_ml_pref();
    assert_macro (n == 0, "empty vector expected, vec(" << n << ") found");
    if (ml_pref.pr_as_read_syntax) {
        os << "[]";
    } else {
      	os << "[]";
      	if (ml_pref.print_empty_dimensions)
	    os << "(" << n << ")";
      	os << "\n";
    }
}
template <class Vec>
static void
print_matlab_gen (ostream& os, const Vec& x)
{
        init_ml_pref();
	int n = x.n();
  	if (n == 0) {

    	    print_empty_vector (os, n);

  	} else if (ml_pref.plus_format && ! ml_pref.pr_as_read_syntax) {

	     for (int i = 0; i < n; i++) {
	         os << "  ";
	         if (x (i) == Float(0.))
		     os << " ";
	      	 else
		     os << "+";
	  	 os << "\n";
	     }
    	} else {
      	    int fw = 0;
      	    set_format (os, x, fw);
      	    int column_width = fw + 2;
      	    int total_width = n * column_width;
      	    int max_width = terminal_columns ();

      	    if (ml_pref.pr_as_read_syntax)
	    	max_width -= 4;

      	    if (ml_pref.free_format) {

	        if (ml_pref.pr_as_read_syntax)
	    	    os << "[\n";

		// default: dump
	        os << x;

	        if (ml_pref.pr_as_read_syntax)
	            os << "]";
	        return;
	    }
	    int inc = n;
            if (total_width > max_width && ml_pref.split_long_rows) {
	  
		inc = max_width / column_width;
		if (inc == 0)
	    	    inc++;
	    }
	    if (ml_pref.pr_as_read_syntax) {

		// x = [ 1; 2; 3; ...
		//       4; 5]
		int col = 0;
		os << "[ ";
	        while (col < n) {
		  
		    int lim = col + inc < n ? col + inc : n;
		    for (int i = col; i < lim; i++) {
		      
			if (i > col) os << "; ";
			pr_float (os, x(i));
		   }
		   col += inc;
		   if (col < n) os << "; ...\n";
		}
		os << " ]";
	    } else {
	  	for (int line = 0; line < n; line += inc) {
	      
		    int lim = line + inc < n ? line + inc : n;
		    if (total_width > max_width && ml_pref.split_long_rows) {
		  
			if (line != 0)
		    	    os << "\n";

		  	int num_lines = lim - line;
		  	if (num_lines == 1)
		    	    os << " Index " << line + 1 << ":\n\n";
		  	else if (num_lines == 2)
		    	    os << " Indexes " << line + 1 << " and " << lim << ":\n\n";
		  	else
		    	    os << " Indexes " << line + 1 << " through " << lim << ":\n\n";
		    }
		    for (int i = line; i < lim; i++) {
		      
		        os << "  ";
			pr_float (os, x(i), fw);
		    }
	        }
		os << "\n";
	    }
        }
}
template <class T>
void
print_matlab (ostream& os, const vec<T>& x)
{
    print_matlab_gen(os, x);
}
template <class T>
void
print_matlab (ostream& os, const avec<T>& x)
{
    print_matlab_gen(os, x);
}
template <class T>
void
print_matlab (ostream& os, const csr<T>& a)
{
        init_ml_pref();
	int nr = a.nrow ();
  	int nc = a.ncol ();
  	if (nr == 0 || nc == 0) {

	    os << "zeros (" << a.nrow() << ", " << a.ncol() << ")";

  	} else if (ml_pref.plus_format && ! ml_pref.pr_as_read_syntax) {
      
	     for (int i = 0; i < nr; i++) {
	         for (int j = 0; j < nc; j++) {
	             if (j == 0)
		        os << "  ";
	             if (a (i, j) == Float(0))
		        os << " ";
	      	     else
		    	os << "+";
	    	 }
	  	 os << "\n";
	     }
    	} else {
      	    int fw = 0;
      	    set_format (os, a, fw);
      	    int column_width = fw + 2;
      	    int total_width = nc * column_width;
      	    int max_width = terminal_columns ();

      	    if (ml_pref.pr_as_read_syntax)
	   	max_width -= 4;

      	    if (ml_pref.free_format) {

	        if (ml_pref.pr_as_read_syntax)
	    	    os << "[\n";

		// default: dump
	        os << a;

	        if (ml_pref.pr_as_read_syntax)
	            os << "]";
	        return;
	    }
	    int inc = nc;
            if (total_width > max_width && ml_pref.split_long_rows) {
	  
		inc = max_width / column_width;
		if (inc == 0)
	    	    inc++;
	    }
	    if (ml_pref.pr_as_read_syntax) {

	  	for (int i = 0; i < nr; i++) {
	      
		    int col = 0;
	            while (col < nc) {
		  
			int lim = col + inc < nc ? col + inc : nc;
			for (int j = col; j < lim; j++) {
		      
			    if (i == 0 && j == 0) {
				os << "[ ";
		      	    } else {
			        if (j > col && j < lim)
			    	    os << ", ";
			  	else
			    	    os << "  ";
			    }
			    pr_float (os, a(i, j));
		    	}
			col += inc;
			if (col >= nc) {
		      
			    if (i == nr - 1)
				os << " ]";
		      	    else
				os << ";\n";
		    	} else {
		    		os << " ...\n";
			}
		    }
		}
	    } else {

	  	for (int col = 0; col < nc; col += inc) {
	      
		    int lim = col + inc < nc ? col + inc : nc;
		    if (total_width > max_width && ml_pref.split_long_rows) {
		  
			if (col != 0)
		    	    os << "\n";

		  	int num_cols = lim - col;
		  	if (num_cols == 1)
		    	    os << " Column " << col + 1 << ":\n\n";
		  	else if (num_cols == 2)
		    	    os << " Columns " << col + 1 << " and " << lim << ":\n\n";
		  	else
		    	    os << " Columns " << col + 1 << " through " << lim << ":\n\n";
		    }
		    for (int i = 0; i < nr; i++) {
		        for (int j = col; j < lim; j++) {
		      
			    os << "  ";
			    pr_float (os, a(i, j), fw);
		        }
			os << "\n";
		    }
	        }
	    }
        }
}
// =====================[ INSTANCIATION IN LIBRARY ]=============================

template void print_matlab (ostream&, const vec<Float>&);
template void print_matlab (ostream&, const avec<Float>&);
template void print_matlab (ostream&, const csr<Float>&);

}// namespace rheolef
