#ifndef NCBI_NK_UNIT_TEST__SUITE_HPP
#define NCBI_NK_UNIT_TEST__SUITE_HPP

#include <iostream> // cerr
#include <sstream>  // ostringstream
#include <vector>
#include <csignal>  // sigaction 
#include <cassert>

////////////////////////////////////////////////////////////////////////////////

namespace ncbi { namespace NK { 

typedef int counter_t; 

class Empty {};

class execution_aborted {};

class LogLevel {
public:
    enum E {
        e_undefined,
        e_all,
        e_test_suite,
        e_message,
        e_warning,
        e_error,
        e_fatal_error,
        e_nothing
    };
};

template<class T> const T abs(const T& a) { return a >= 0 ? a : -a; }

class TestEnv {
public:
    TestEnv(int argc, char* argv[]) : catch_system_errors(true) {
        process_args(argc, argv);
    }

    void set_handlers(void) {
        struct sigaction act;
        memset(&act, 0, sizeof act);
        act.sa_handler = SigHandler;
        act.sa_flags = SA_RESETHAND;
        int status = sigaction(SIGFPE , &act, NULL);
        status     = sigaction(SIGILL , &act, NULL);
        status     = sigaction(SIGSEGV, &act, NULL);
    }

    static std::string lastLocation;
    static LogLevel::E verbosity;
    bool catch_system_errors;

private:
    static void SigHandler(int sig) {
        switch (sig) {
            case SIGFPE:
                LOG(ncbi::NK::LogLevel::e_fatal_error,
                    "fatal error: signal: SIGFPE(Floating point exception)\n");
                break;
            case SIGILL:
                LOG(ncbi::NK::LogLevel::e_fatal_error,
                    "fatal error: signal: SIGILL(Illegal instruction)\n");
                break;
            case SIGSEGV:
                LOG(ncbi::NK::LogLevel::e_fatal_error,
                    "fatal error: signal: SIGSEGV(Segmentation fault)\n");
                break;
            default:
                LOG(ncbi::NK::LogLevel::e_fatal_error,
                    "fatal error: signal: " << sig << "\n");
                break;
        }
        if (!ncbi::NK::TestEnv::lastLocation.empty()) {
            LOG(ncbi::NK::LogLevel::e_fatal_error,
                ncbi::NK::TestEnv::lastLocation << ": last checkpoint\n");
        }
        LOG(ncbi::NK::LogLevel::e_fatal_error, "Test is aborted\n");
        exit(1);
    }

    void process_args(int argc, char* argv[]) {
        bool debug = false;
        LogLevel::E detected = LogLevel::e_undefined;
        char arg_catch_system_errors[] = "-catch_system_errors=";
        char arg_log_level          [] = "-l=";
        for (int i = 1; i < argc; ++i) {
            if (strncmp(argv[i], arg_log_level, strlen(arg_log_level))
                == 0)
            {
                char* a = argv[i] + strlen(arg_log_level);
                if (strcmp(a, "test_suite") == 0)
                { detected = LogLevel::e_test_suite; }
                else if (strcmp(a, "all"    ) == 0)
                { detected = LogLevel::e_all; }
                else if (strcmp(a, "message") == 0)
                { detected = LogLevel::e_message; }
                else if (strcmp(a, "warning") == 0)
                { detected = LogLevel::e_warning; }
                else if (strcmp(a, "error"  ) == 0)
                { detected = LogLevel::e_error; }
                else if (strcmp(a, "nothing") == 0)
                { detected = LogLevel::e_nothing; }
                else if (strcmp(a, "fatal_error") == 0)
                { detected = LogLevel::e_fatal_error; }
                if (detected != LogLevel::e_undefined) {
                    verbosity = detected;
                    if (debug) {
                        LOG(LogLevel::e_nothing,
                            "debug: log_level was set to " << a << std::endl);
                    }
                }
                else {
                    verbosity = LogLevel::e_error;
                    if (debug) {
                        LOG(LogLevel::e_nothing,
                            "debug: log_level was set to error\n");
                    }
                }
            }
            else if (strncmp(argv[i], arg_catch_system_errors,
                strlen(arg_catch_system_errors)) == 0)
            {
                char* a = argv[i] + strlen(arg_catch_system_errors);
                if (strcmp(a, "n") == 0 || strcmp(a, "no") == 0) {
                    catch_system_errors = false;
                    if (debug) {
                        LOG(LogLevel::e_nothing,
                           "debug: arg_catch_system_errors was set to false\n");
                    }
                }
                else {
                    if (debug) {
                        LOG(LogLevel::e_nothing,
                           "debug: arg_catch_system_errors was set to true\n");
                    }
                }
            }
            else if (strcmp(argv[i], "-debug") == 0) {
                debug = true;
                LOG(LogLevel::e_nothing, "debug: debug was set to true\n");
            }
            else if (strcmp(argv[i], "-h") == 0) {
                if (debug) {
                    LOG(LogLevel::e_nothing, "debug: help was set to true\n");
                }
                std::cerr << "Usage:\n"
<< argv[0] << " [-debug] [-catch_system_errors=[yes|y|no|n]] "
"[-l=<value>] [-h]\nwhere:\n"
"debug - Print recognized command line arguments (should be specified first)\n"
"catch_system_errors - "
  "Allows to switch between catching and ignoring system errors (signals)\n"
"l (log_level) - Specifies log level\n"
"\tall        - report all log messages\n"
"\t             including the passed test notification\n"
"\ttest_suite - show test suite messages\n"
"\tmessage    - show user messages\n"
"\twarning    - report warnings issued by user\n"
"\terror      - report all error conditions (default)\n"
"\tfatal_error- report user or system originated fatal errors\n"
"\t             (for example, memory access violation)\n"
"\tnothing    - do not report any information\n"
"h (help) - this help message\n"; exit(0);
            }
        }

        if (verbosity == LogLevel::e_undefined) {
            verbosity = LogLevel::e_error;
            if (debug) {
                LOG(LogLevel::e_nothing,
                    "debug: log_level was set to error\n");
            }
        }
    }
};

class TestCase {
protected:
    TestCase(const std::string& name) : _name(name), _ec(0) {}

public:
    ncbi::NK::counter_t GetErrorCounter(void) { return _ec; }
    const std::string& GetName(void) const { return _name; }

protected:
    void report_error(const char* msg, const char* file, int line,
        bool is_msg = false, bool isCritical = false)
    {
        ncbi_NK_saveLocation(file, line);
        ++_ec;
        LOG(LogLevel::e_error, file << "(" << line << "): ");
        if (isCritical) {
            LOG(LogLevel::e_error, "fatal ");
        }
        LOG(LogLevel::e_error, "error in \"" << _name << "\": ");
        if (is_msg) {
            LOG(LogLevel::e_error, msg);
        }
        else {
            if (isCritical) {
                LOG(LogLevel::e_error, "critical ");
            }
            LOG(LogLevel::e_error, "check " << msg << " failed");
        }
        LOG(LogLevel::e_error, std::endl);
        if (isCritical)
        { throw ncbi::NK::execution_aborted(); }
    }

    void report_passed(const char* msg, const char* file, int line) {
        ncbi_NK_saveLocation(file, line);
        LOG(LogLevel::e_all, file << "(" << line << "): info: "
            "check " << msg << " passed" << std::endl);
    }

    template<class T1, class T2>
    void report_error2(const char* e1, const char* e2, T1 t1, T2 t2,
        const char* file, int line, const char* eq, const char* ne,
        bool isCritical = false)
    {
        ncbi_NK_saveLocation(file, line);
        ++_ec;
        LOG(LogLevel::e_error, file << "(" << line << "): ");
        if (isCritical) {
            LOG(LogLevel::e_error, "fatal ");
        }
        LOG(LogLevel::e_error, "error in \"" << _name << "\": ");
        if (isCritical) {
            LOG(LogLevel::e_error, "critical ");
        }
        LOG(LogLevel::e_error, "check " << e1 << " " << eq << " " << e2
            << " failed [" << t1 << " " << ne << " " << t2 << "]");
        LOG(LogLevel::e_error, std::endl);
        if (isCritical)
        { throw ncbi::NK::execution_aborted(); }
    }

    template<class T1, class T2>
    void report_passed2(const char* e1, const char* e2,
        const T1& t1, const T2& t2,
        const char* file, int line, const char* eq, const char* ne)
    {
        ncbi_NK_saveLocation(file, line);
        LOG(LogLevel::e_all, file << "(" << line << "): info: "
            "check " << e1 << " " << eq << " " << e2 << " passed" << std::endl);
    }

    template<class T1, class T2, class T3>
    void report_passed_close(const char* e1, const char* e2,
        const T1& t1, const T2& t2, const T3& tolerance,
        const char* file, int line)
    {
        ncbi_NK_saveLocation(file, line);
        LOG(LogLevel::e_all, file << "(" << line << "): "
            "info: difference between "
            << e1 << "{" << t1 << "} and " << e2 << "{" << t2 << "} "
            "doesn't exceed " << tolerance << std::endl);
    }

    template<class T1, class T2, class T3, class T4>
    void report_error_close(const char* e1, const char* e2,
        const T1& t1, const T2& t2, const T3& tolerance, const T4& diff,
        const char* file, int line, bool isCritical = false)
    {
        ncbi_NK_saveLocation(file, line);
        ++_ec;
        LOG(LogLevel::e_error, file << "(" << line << "): ");
        if (isCritical) {
            LOG(LogLevel::e_error, "fatal ");
        }
        LOG(LogLevel::e_error, "error in \"" << _name << "\": "
            "difference{" << diff << "} between "
            << e1 << "{" << t1 << "} and " << e2 << "{" << t2 << "} "
            "exceeds " << tolerance << std::endl);
        if (isCritical)
        { throw ncbi::NK::execution_aborted(); }
    }

    void ncbi_NK_saveLocation(const char* file, int line) {
        std::ostringstream s;
        s << file << "(" << line << ")";
        ncbi::NK::TestEnv::lastLocation = s.str();
    }

    void _REPORT_CRITICAL_ERROR_(const std::string& msg,
        const char* file, int line, bool is_msg = false)
    { report_error( msg.c_str(), file, line, is_msg, true ); }

private:
    const std::string _name;
    ncbi::NK::counter_t _ec;
};

class TestInvoker {
protected:
    TestInvoker(const std::string& name) : _name(name), _ec(0) {}
    virtual ~TestInvoker(void) {}
public:
    virtual void Run(void* globalFixtute) = 0;
    const std::string& GetName(void) const { return _name; }
    ncbi::NK::counter_t GetErrorCounter(void) { return _ec; }
protected:
    void SetErrorCounter(ncbi::NK::counter_t ec) { _ec = ec; }
private:
    const std::string _name;
    ncbi::NK::counter_t _ec;
};

class TestRunner {
    typedef std::vector<ncbi::NK::TestInvoker*> T;
    typedef T::const_iterator TCI;

public:
    TestRunner() : argc(0), argv(NULL) {}

    int    argc;
    char** argv;

    void ReportTestNumber(void) {
        T::size_type sz = _cases.size();
        if (sz == 1) {
           LOG(LogLevel::e_fatal_error, "Running " << sz << " test case...\n");
        } else if (sz > 1) {
           LOG(LogLevel::e_fatal_error, "Running " << sz << " test cases...\n");
        }
    }
    void SetArgs(int argc, char* argv[]) {
        this->argc = argc;
        this->argv = argv;
    }
    void Add(ncbi::NK::TestInvoker* t) {
        if (t) {
            _cases.push_back(t);
        }
    }
    ncbi::NK::counter_t Run(void* globalFixtute) const {
        ncbi::NK::counter_t ec = 0;
        for (TCI it = _cases.begin(); it != _cases.end(); ++it) {
           ncbi::NK::TestInvoker* c = *it;
           try {
               LOG(LogLevel::e_test_suite,
                   "Entering test case \"" << c->GetName() << "\"\n");
               c->Run(globalFixtute);
               LOG(LogLevel::e_test_suite,
                   "Leaving test case \"" << c->GetName() << "\"\n");
           } catch (const ncbi::NK::execution_aborted&) {
               LOG(LogLevel::e_test_suite,
                   "Leaving test case \"" << c->GetName() << "\"\n");
           }
           ec += c->GetErrorCounter();
        }
        return ec;
    }

private:
    T _cases;
};

static ncbi::NK::TestRunner* GetTestSuite(void)
{ static ncbi::NK::TestRunner* t = new ncbi::NK::TestRunner; return t; }

template<class TFixture>
ncbi::NK::counter_t Main(int argc, char* argv[],
                         const char* suite_name)
{
    ncbi::NK::counter_t ec = 0;
    ncbi::NK::TestRunner* t = ncbi::NK::GetTestSuite();
    assert(t);
    t->SetArgs(argc, argv);
    t->ReportTestNumber();
    try {
        TFixture globalFixtute;
        LOG(ncbi::NK::LogLevel::e_test_suite,
            "Entering test suite \"" << suite_name << "\"\n");
        ec = t->Run(&globalFixtute);
        LOG(ncbi::NK::LogLevel::e_test_suite,
            "Leaving test suite \"" << suite_name << "\"\n";)
    } catch (const ncbi::NK::execution_aborted& e) { ++ec; }
    switch (ec) {
        case 0:
          LOG(ncbi::NK::LogLevel::e_nothing, "\n*** No errors detected\n");
          break;
        case 1:
          LOG(ncbi::NK::LogLevel::e_nothing, "\n*** " << ec <<
           " failure detected in test suite \"" << suite_name << "\"\n");
          break;
        default:
          LOG(ncbi::NK::LogLevel::e_nothing, "\n*** " << ec <<
           " failures detected in test suite \"" << suite_name << "\"\n");
          break;
    }
    return ec;
}

} } // namespace ncbi::NK

inline void ncbi_NK_saveLocation(const char* file, int line) {}
inline void _REPORT_CRITICAL_ERROR_(const std::string& msg,
                                    const char* file, int line, bool is_msg)
{
    LOG(ncbi::NK::LogLevel::e_error, file << "(" << line << "): ");
    LOG(ncbi::NK::LogLevel::e_error, "fatal ");
    LOG(ncbi::NK::LogLevel::e_error, "error in fixture: ");
    if (is_msg) {
        LOG(ncbi::NK::LogLevel::e_error, msg);
    }
    else {
        LOG(ncbi::NK::LogLevel::e_error, "critical ");
        LOG(ncbi::NK::LogLevel::e_error, "check " << msg << " failed");
    }
    LOG(ncbi::NK::LogLevel::e_error, std::endl);
    throw ncbi::NK::execution_aborted();
 }

#endif// NCBI_NK_UNIT_TEST__SUITE_HPP
